<?php

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Exceptions\RunnerException;
use DefStudio\GameEngine\Decorators\Runs\PlayableTask;

/**
 * @property-read PlayableTask $task
 */
trait RunsTasks
{
    protected function is_task_active(): bool
    {
        return $this->current_task_index() >= 0;
    }

    protected function current_task_index(): int
    {
        return $this->run->get_state('task.index', -1);
    }

    public function getTaskProperty(): PlayableTask
    {
        return $this->mission->playable_tasks()
            ->values()
            ->get($this->current_task_index()) ?? throw RunnerException::no_active_task();
    }

    protected function next_task(): void
    {
        if ($this->is_task_active()) {
            throw_unless($this->task->completed(), RunnerException::incomplete_task($this->task));

            if ($this->current_task_index() >= $this->mission->playable_tasks()->count() - 1) {
                $this->complete_mission();

                return;
            }
        }

        $this->activate_task($this->current_task_index() + 1);

        $this->task->start_timer();
        $this->run->save();
    }

    protected function activate_task(int $index): void
    {
        $this->reset_task_state();

        $this->run->set_state('task.index', $index);

        $this->forgetComputed('task');

        $this->setup_storytellings($this->task, true);

        $this->show_storytelling($this->task, StorytellingType::activation);

        $this->run->save();
    }

    protected function reset_task_state(): void
    {
        $this->run->set_state('task', ['index' => -1]);
        $this->reset_storytellings('task');
        $this->forgetComputed('task');
    }

    public function get_task_state(string $key, mixed $default = null): mixed
    {
        return $this->task->get_state($key, $default);
    }

    public function set_task_state(string $key, mixed $value, bool $save = true): void
    {
        $this->task->set_state($key, $value);

        if ($save) {
            $this->run->save();
        }
    }

    public function complete_task_with_result(array $result): void
    {
        $state = $this->task->get_state();
        $state['result'] = $result;
        $state = $this->task->process_completion($state);
        $this->run->set_state("tasks.{$this->task->id}", $state);

        $this->run->save();

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->assign_awards($this->task);
        }

        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->assign_level($this->task);
        }

        $this->task->stop_timer();
        $this->run->save();

        $this->next_task();
    }
}
