<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Users;

use DefStudio\GameEngine\Enums\Role;
use DefStudio\GameEngine\Models\User;
use DefStudio\WiredTables\WiredTable;
use Illuminate\Database\Eloquent\Builder;
use DefStudio\GameEngine\Actions\Users\DeleteUser;
use Illuminate\Database\Eloquent\Relations\Relation;

class Table extends WiredTable
{
    protected $listeners = [
        'user.created' => '$refresh',
        'user.saved' => '$refresh',
        'user.updated' => '$refresh',
    ];

    protected function query(): Builder|Relation
    {
        $query = match (true) {
            \game_engine()->role(Role::super_admin) => \game_engine()->userQuery(),
            !!\game_engine()->user()->current_organization => \game_engine()->user()->current_organization->users(),
            default => \game_engine()->userQuery()->whereHas('organizations', function(Builder $subquery): void {
                $subquery->whereIn('organization_id', \game_engine()->user()->organizations()->pluck('id'));
            }),
        };

        return $query->with('organizations');
    }

    protected function filters(): void
    {
        $organization_query = \game_engine()->role(Role::super_admin)
            ? \game_engine()->organizationQuery()
            : \game_engine()->user()->organizations();

        $this->filter(trans_choice('Organization|Organizations', 1), 'organization_id')
            ->select($organization_query->pluck('name', 'id')->toArray())
            ->handle(fn(Builder $query, int $organization_id) => $query->whereHas('organizations', fn(Builder $subquery) => $subquery->where('organization_id', $organization_id)));
    }

    protected function columns(): void
    {
        $this->column(__('Name'), 'name')->searchable()->sortable();
        $this->column(__('Email'), 'email')->searchable()->sortable();
        $this->column(trans_choice('Organization|Organizations', 2))->view('game-engine::livewire.admin.users.table.role_list');
        $this->column(trans_choice('Action|Actions', 2))->view('game-engine::livewire.admin.users.table.actions');
    }

    public function delete(User $user): void
    {
        DeleteUser::run($user);
    }
}
