<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Actions\Users;

use Illuminate\Support\Str;
use DefStudio\Actions\Action;
use DefStudio\GameEngine\Enums\Role;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Hash;
use DefStudio\GameEngine\Models\User;
use DefStudio\GameEngine\Events\Users\UserSaved;
use DefStudio\GameEngine\Events\Users\UserCreated;
use DefStudio\GameEngine\Events\Users\UserUpdated;

/**
 * @method static User run(User $user, Role $role = null, string $password = '', bool $authorize = true)
 */
class SaveUser extends Action
{
    public function handle(User $user, Role $role = null, string $password = '', bool $authorize = true): User
    {
        if ($authorize) {
            $this->authorize($user, $role);
        }

        if (!$user->exists) {
            if ($password !== '') {
                $user->password = Hash::make($password);
            } elseif (config('game-engine.users.assign_password_on_creation', true)) {
                $password = Str::password();
                $user->password = Hash::make($password);
            }

            $user->save();

            UserCreated::dispatch($user, empty($password) ? null : $password);
        } else {
            if ($password !== '') {
                $user->password = Hash::make($password);
            }

            $user->save();

            UserUpdated::dispatch($user);
        }

        UserSaved::dispatch($user);

        return $user;
    }

    public function authorize(User $user, Role $role = null): void
    {
        if ($user->exists) {
            Gate::authorize('update', $user);
        } elseif ($role === null) {
            Gate::authorize('create', game_engine()->userClass());
        } else {
            Gate::authorize('create', $role);
        }
    }
}
