<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Exceptions\RunnerException;
use DefStudio\GameEngine\Decorators\Runs\PlayableMission;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

trait RunsMissions
{
    protected function start_runs_missions(): void
    {
        if (!$this->is_mission_active()) {
            return;
        }

        $this->mission->start_timer();

        $this->save();

        if (!$this->show_storytelling($this->mission, StorytellingType::activation)) {
            if ($this->mission->should_autostart_on_activation()) {
                $this->start_mission($this->current_mission_index());
            }
        }
    }

    public function get_missions(): array
    {
        if (!$this->is_map_active()) {
            return [];
        }

        return $this->map->playable_missions()->map(fn(PlayableMission $mission): array => [
            'id' => $mission->id,
            'placeholder_image' => $mission->current_placeholder(),
            'pos_x' => $mission->pos_x,
            'pos_y' => $mission->pos_y,
            'visible' => $mission->visible(),
            'always_active' => $mission->is_always_active() && !$mission->completed(),
        ])->toArray();
    }

    protected function reset_mission_state(): void
    {
        $this->run->set_state('current_mission', -1);
        $this->run->set_state('current_task', -1);

        $this->forgetComputed('mission');
    }

    public function next_mission(bool $force = false): void
    {
        if ($this->is_mission_active()) {
            throw_unless($force || $this->mission->completed(), RunnerException::incomplete_mission($this->mission));
        }

        if ($this->map->playable_missions()->every(fn(PlayableMission $mission) => $mission->completed())) {
            $this->complete_map();

            return;
        }

        $first_available_mission = $this->map->playable_missions()->sortBy('order')->values()
            ->filter(fn(PlayableMission $mission) => !$mission->completed())
            ->keys()->first();

        $this->activate_mission($first_available_mission);
        $this->save();
    }

    protected function activate_mission(int $index): void
    {
        $this->reset_mission_state();

        $this->run->set_state('current_mission', $index);

        if ($this->mission->get_state('extracted_tasks') === null) {
            $extracted = game_engine()->task_extractor()->extract($this->mission);

            $this->run->set_state("missions.{$this->mission->id}.extracted_tasks", $extracted->pluck('id')->toArray());
        }

        $this->forgetComputed('mission');

        $this->emit('mission:activated');

        $this->mission->setup_storytelling();

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->compute_starting_awards($this->mission);
        }

        $this->save();

        if ($this->show_storytelling($this->mission, StorytellingType::activation)) {
            return;
        }

        $this->mission->set_state('storytelling.'.StorytellingType::activation->value, true);

        if ($this->mission->should_complete_on_activation()) {
            $this->complete_mission();

            return;
        }

        $this->save();

        if ($this->mission->should_autostart_on_activation()) {
            $this->start_mission($this->current_mission_index());
        }
    }

    public function start_mission(int $index): void
    {
        $intended_mission = $this->map->playable_missions()->sortBy('order')->values()->get($index);

        if ($this->current_mission_index() !== $index && !$intended_mission->is_always_active()) {
            throw RunnerException::mission_not_active($intended_mission);
        }

        if ($this->mission->id !== $intended_mission->id) {
            $this->activate_mission($index);
        }

        if ($this->mission->all_tasks_completed()) {
            $this->complete_mission();

            return;
        }

        if (!$this->show_storytelling($this->mission, StorytellingType::prologue)) {
            $this->open_mission_modal();
        }
    }

    protected function complete_storytelling_runs_missions(PlayableActivity $activity, StorytellingType $type): void
    {
        if (!($activity instanceof PlayableMission)) {
            return;
        }

        if ($type === StorytellingType::activation) {
            if ($this->mission->should_complete_on_activation()) {
                $this->complete_mission();
            }

            return;
        }

        if ($type === StorytellingType::prologue) {
            $this->open_mission_modal();

            return;
        }

        if ($type === StorytellingType::epilogue) {
            $this->complete_mission();
        }

        if ($type === StorytellingType::pause) {
            if ($this->mission->get_state('timed_out')) {
                $this->complete_mission(StorytellingType::timeout->value, true);
            } else {
                $this->complete_map();
            }
        }

        if ($type === StorytellingType::timeout) {
            $this->complete_mission(StorytellingType::timeout->value, true);
        }
    }

    protected function open_mission_modal(): void
    {
        $this->mission->start_timer();
        $this->save();
        $this->open_modal('game-engine.runner.mission', [
            'run_id' => $this->run->id,

            /*
             * This is not used but needed because wire-modals can't open a
             * component twice with the same params. If missing, when
             * there are two subsequent missions, it will not open
             */
            'mission_id' => $this->mission->id,
        ]);
    }

    public function complete_mission(string $storytelling_type = null, bool $force = false): void
    {
        $storytelling_type = StorytellingType::from($storytelling_type ?? StorytellingType::epilogue->value);

        throw_unless($force || $this->mission->all_tasks_completed(), RunnerException::incomplete_mission($this->mission));

        $this->mission->stop_timer();

        if ($storytelling_type === StorytellingType::timeout) {
            $this->mission->set_state('timed_out', true);
        }

        $this->save();

        if ($this->show_storytelling($this->mission, $storytelling_type)) {
            return;
        }

        if ($this->show_storytelling($this->mission, StorytellingType::pause)) {
            $this->mission->pause();

            return;
        }

        $this->mission->set_state('completed', true);
        $this->run->save();

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->assign_awards($this->mission);
        }

        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->assign_level($this->mission);
        }

        $this->save();

        $this->emit('mission:completed');

        $this->next_mission($force);

        $this->close_modal();
    }
}
