<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\TimeUnit;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Exceptions\RunnerException;
use DefStudio\GameEngine\Decorators\Runs\PlayableTask;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

trait RunsTasks
{
    protected function next_task(bool $force = false): void
    {
        if ($this->is_task_active()) {
            throw_unless($force || $this->task->completed(), RunnerException::incomplete_task($this->task));

            if ($this->current_task_index() >= $this->mission->playable_tasks()->count() - 1) {
                $this->complete_mission();

                return;
            }
        }

        $this->activate_task($this->current_task_index() + 1);

        $this->task->start_timer();
        $this->run->save();
    }

    protected function activate_task(int $index): void
    {
        $this->reset_current_state();

        $this->run->set_state('current_task', $index);

        $this->forgetComputed('task');

        $this->task->setup_storytelling();

        $this->show_storytelling($this->task, StorytellingType::activation, StorytellingType::prologue);

        $this->run->save();

        $this->emit('task:activated');
    }

    protected function reset_current_state(): void
    {
        $this->run->set_state('current_task', -1);
        $this->forgetComputed('task');
    }

    protected function complete_storytelling_runs_tasks(PlayableActivity $activity, StorytellingType $type): void
    {
        if (!($activity instanceof PlayableTask)) {
            return;
        }

        if ($type === StorytellingType::epilogue) {
            $this->next_task();
        }

        if ($type === StorytellingType::timeout) {
            $this->next_task(true);
        }
    }

    public function complete_task_with_result(array $result, string $storytelling_type = null, bool $force = false): void
    {
        $storytelling_type = StorytellingType::from($storytelling_type ?? StorytellingType::epilogue->value);

        $state = $this->task->get_state();
        $state['result'] = $result;
        $state = $this->task->process_completion($state);

        $this->task->replace_state($state);

        $this->task->stop_timer();
        $this->run->save();

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->assign_awards($this->task);
        }

        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->assign_level($this->task);
        }

        $this->task->stop_timer();
        $this->run->save();

        if ($this->show_storytelling($this->task, $storytelling_type)) {
            return;
        }

        $this->next_task($force);
    }

    public function get_task_deadline(): ?string
    {
        if (Feature::deadlines->disabled()) {
            return null;
        }

        if (empty($this->task->configuration['time_limit'])) {
            return null;
        }

        $elapsed_seconds = $this->task->get_elapsed_time()->totalSeconds;

        $remaining_seconds = TimeUnit::from($this->task->configuration['time_limit_unit'])->to_seconds($this->task->configuration['time_limit']) - $elapsed_seconds;

        return now()->addSeconds($remaining_seconds)->toIso8601String();
    }

    public function force_task_timeout(): void
    {
        if (empty($this->task->configuration['time_limit'])) {
            return;
        }

        $this->complete_task_with_result([], StorytellingType::timeout->value, true);
    }
}
