<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

/** @noinspection PhpVoidFunctionResultUsedInspection */

namespace DefStudio\GameEngine\Http\Livewire\Runs;

use Carbon\Carbon;
use DefStudio\GameEngine\Enums\Role;
use DefStudio\GameEngine\Models\Run;
use Illuminate\Support\Facades\Gate;
use DefStudio\WiredTables\WiredTable;
use DefStudio\WiredTables\Enums\Config;
use Illuminate\Database\Eloquent\Builder;
use DefStudio\GameEngine\Models\Organization;
use Symfony\Component\HttpFoundation\Response;
use DefStudio\GameEngine\Actions\Runs\DeleteRun;
use Illuminate\Database\Eloquent\Relations\Relation;
use DefStudio\WiredTables\Configurations\TableConfiguration;

class Table extends WiredTable
{
    public function mount(): void
    {
        if (empty($this->filterValues['status'])) {
            $this->filterValues['status'] = 'completed';
        }
    }

    protected function user_is_super_admin(): bool
    {
        return game_engine()->role(Role::super_admin);
    }

    protected function configure(TableConfiguration $configuration): void
    {
        $configuration
            ->pageSize(10)
            ->groupFilters(false)
            ->footerView('game-engine::livewire.admin.runs.table.footer');
    }

    protected function query(): Builder|Relation
    {
        $query = match (true) {
            $this->user_is_super_admin() => game_engine()->runQuery(),
            $this->user_can_view_organizations() => game_engine()->user()->current_organization->runs(),
            $this->user_can_view_runs() => $this->user_teams_runs_query(),

            default => abort(Response::HTTP_FORBIDDEN),
        };

        return $query
            ->where('debug', 0)
            ->with('user', 'story');
    }

    protected function filters(): void
    {
        $this->filter(__('Status'), 'status')
            ->select(['completed' => __('Show Completed'), 'ongoing' => __('Show Ongoing')])
            ->placeholder(__('Show All Runs'))
            ->handle(fn(Builder|Relation $query, string $status) => match ($status) { //@phpstan-ignore-line
                'completed' => $query->completed(), //@phpstan-ignore-line
                'ongoing' => $query->ongoing(), //@phpstan-ignore-line
            });

    }

    protected function columns(): void
    {
        $this->column(trans_choice('User|Users', 1), 'user.name')->searchable()->sortable();
        $this->column(trans_choice('Story|Stories', 1), 'story.title')->searchable()->sortable();
        $this->column(__('Started'), 'started_at')->format(fn(?Carbon $started) => $started?->format('d/m/Y H:i'));
        $this->column(__('Status'))->view('game-engine::livewire.admin.runs.table.status');
        $this->column(trans_choice('Action|Actions', 2))->view('game-engine::livewire.admin.runs.table.actions');
    }

    protected function user_can_view_organizations(): bool
    {
        $user = game_engine()->user();

        if (empty($user->current_organization_id)) {
            return false;
        }

        if (!Gate::check('viewAny', Organization::class)) {
            return false;
        }

        return Gate::check('viewAny', Run::class);
    }

    protected function user_can_view_runs(): bool
    {
        $user = game_engine()->user();

        if (empty($user->current_organization_id)) {
            return false;
        }

        return Gate::check('viewAny', Run::class);
    }

    /**
     * @return Relation<Run>
     */
    protected function user_teams_runs_query(): Builder
    {
        $user = game_engine()->user()->loadMissing('teams', 'current_teams');

        return $user
            ->current_organization->runs()
            ->where(fn(Builder $query) => $query
                ->whereRelation(
                    'user',
                    fn(Builder $user_query) => $user_query->whereHas('teams', fn(Builder $team_query) => $team_query->whereIn('teams.id', $user->current_teams()->pluck('id'))
                    )
                ));
    }

    protected function applyRowsSelection(\Illuminate\Contracts\Database\Eloquent\Builder|Relation $query): void
    {
        if (empty($this->selection)) {
            return;
        }

        $query->whereIn(config('game-engine.runs.table').'.'.$this->config(Config::id_field), $this->selectedIds());
    }

    public function delete(int $run_id): void
    {
        /** @var Run $run */
        $run = game_engine()->runQuery()->findOrFail($run_id);

        DeleteRun::run($run);
    }

    public function resume(): void
    {
        $runs = empty($this->selectedIds())
            ? $this->rows
            : $this->selectedRows;

        $runs->each(fn(Run $run) => $run->resume()->save());

        $this->unselectAllRows();
    }
}
