<?php

/** @noinspection PhpUndefinedMethodInspection */

namespace DefStudio\GameEngine\Http\Livewire\Teams;

use Livewire\Component;
use Illuminate\Contracts\View\View;
use DefStudio\GameEngine\Enums\Role;
use Illuminate\Support\Facades\Auth;
use DefStudio\GameEngine\Models\Team;
use DefStudio\GameEngine\Models\User;
use Illuminate\Support\Facades\Route;
use Illuminate\Validation\Rules\Password;
use DefStudio\GameEngine\Models\Organization;
use DefStudio\GameEngine\Actions\Teams\SaveTeam;
use DefStudio\GameEngine\Actions\Users\SaveUser;

class Register extends Component
{
    /** @var Organization */
    public $organization;

    public string $master_name = '';

    public string $master_email = '';

    public string $master_password = '';

    public bool $terms = false;

    public string $team_name = '';

    public bool $auto_assign_team_name = false;

    public string $team_name_prefix = '';

    public function rules(): array
    {
        $rules = [
            'master_name' => ['required'],
            'master_email' => 'required|email|unique:users,email',
            'master_password' => ['required', $this->password_rules()],
        ];

        if (!$this->auto_assign_team_name) {
            $rules['team_name'] = ['required'];
        }

        if (Route::has('policy.show')) {
            $rules['terms'] = 'accepted';
        }

        return $rules;
    }

    protected function password_rules(): Password
    {
        return Password::min(8)->numbers()->letters()->symbols()->uncompromised()->mixedCase();
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.teams.register');
    }

    public function register(): void
    {
        $this->validate();

        $team = $this->make_team();
        SaveTeam::run($team, authorize: false);

        $user = $this->make_user();
        SaveUser::run($user, password: $this->master_password, authorize: false);

        Auth::logout();
        Auth::login($user);

        $user->subscribe_to_organization($this->organization, Role::master, true);
        $user->subscribe_to_team($team);

        $this->after_register($user, $team);

        $this->redirect(\route(config('game-engine.teams.register_team.redirect_to_route')));
    }

    protected function after_register(User $user, Team $team): void
    {
    }

    protected function make_user(): User
    {
        return game_engine()->userClass()::make([
            'name' => $this->master_name,
            'email' => $this->master_email,
        ]);
    }

    public function make_team(): Team
    {
        $name = $this->auto_assign_team_name
            ? str((game_engine()->teamQuery()->max('id') ?? 0) + 1)->padLeft(3, '0')
            : $this->team_name;

        return $this->organization->teams()->make([
            'name' => $this->team_name_prefix.$name,
        ]);
    }
}
