<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */
/** @noinspection PhpUnused */

/** @noinspection LaravelUnknownEloquentFactoryInspection */

namespace DefStudio\GameEngine\Models;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use DefStudio\GameEngine\DTO\Runs\PauseInfo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use DefStudio\GameEngine\Database\Factories\MapFactory;
use DefStudio\GameEngine\Database\Factories\RunFactory;
use DefStudio\GameEngine\Decorators\Runs\PlayableStory;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

use function game_engine;

/**
 * @property int $id;
 * @property int $story_id;
 * @property int $user_id;
 * @property Carbon|null $started_at;
 * @property Carbon|null $completed_at;
 * @property Carbon|null $available_from_date;
 * @property Carbon|null $available_until_date;
 * @property-read Story $story,
 * @property-read User $user,
 * @property-read RunData $data,
 *
 * @method static MapFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
class Run extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'story_id',
        'started_at',
        'completed_at',
        'available_from_date',
        'available_until_date',
        'debug',
    ];

    protected $casts = [
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'debug' => 'bool',
        'available_from_date' => 'date:Y-m-d',
        'available_until_date' => 'date:Y-m-d',
    ];

    protected $attributes = [
        'debug' => false,
    ];

    public function getTable()
    {
        return config('game-engine.runs.table', parent::getTable());
    }

    protected static function newFactory(): RunFactory
    {
        return RunFactory::new();
    }

    protected static function booted(): void
    {
        self::created(function(Run $run) {
            game_engine()->runDataQuery()->create(['run_id' => $run->id]);
        });
    }

    public function scopeOfStory(Builder $query, Story|int $story): void
    {
        $story_id = $story instanceof Story ? $story->id : $story;

        $query->where('story_id', $story_id);
    }

    public function scopeForUser(Builder $query, User|int $user): void
    {
        $user_id = $user instanceof User ? $user->id : $user;

        $query->where('user_id', $user_id);
    }

    public function scopeCompleted(Builder $query): void
    {
        $query->whereNotNull('completed_at');
    }

    public function scopeOngoing(Builder $query): void
    {
        $query->whereNull('completed_at')->whereNotNull('started_at');
    }

    public function scopeActive(Builder $query): void
    {
        $query->whereNull('completed_at');
    }

    /**
     * @return BelongsTo<Story, Run>
     */
    public function story(): BelongsTo
    {
        return $this->belongsTo(game_engine()->storyClass());
    }

    /**
     * @return BelongsTo<User, Run>
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(game_engine()->userClass());
    }

    /**
     * @return HasOne<RunData>
     */
    public function data(): HasOne
    {
        return $this->hasOne(game_engine()->runDataClass());
    }

    public function reset_state(): void
    {
        $this->data->reset_state();
    }

    public function set_state(string $key, mixed $value): void
    {
        $this->data->set_state($key, $value);
    }

    public function forget_state(string $key): void
    {
        $this->data->forget_state($key);
    }

    public function get_state(string $key, mixed $default = null): mixed
    {
        return $this->data->get_state($key, $default);
    }

    public function append_state(string $key, mixed $value): void
    {
        $this->data->append_state($key, $value);
    }

    public function is_paused(): PauseInfo|false
    {
        return $this->data->is_paused();
    }

    public function resume(): self
    {
        $this->data->resume();

        return $this;
    }

    public function pause(PlayableActivity $activity): self
    {
        $this->data->pause($activity);

        return $this;
    }

    public function awards(array $filter_award_ids = null): Collection
    {
        return $this->data->awards($filter_award_ids);
    }

    public function playable_story(): PlayableStory
    {
        return PlayableStory::build($this->story)->setup($this);
    }

    public function save(array $options = []): bool
    {

        if ($this->relationLoaded('data') && $this->data->isDirty()) {
            $this->data->save();
        }

        return parent::save($options);
    }
}
