<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */
/** @noinspection PhpUnused */

/** @noinspection LaravelUnknownEloquentFactoryInspection */

namespace DefStudio\GameEngine\Models;

use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Model;
use DefStudio\GameEngine\DTO\Runs\PauseInfo;
use DefStudio\GameEngine\Exceptions\RunnerException;
use Illuminate\Database\Eloquent\Casts\AsArrayObject;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use DefStudio\GameEngine\Database\Factories\MapFactory;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

use function game_engine;

/**
 * @property int $id;
 * @property array<string, mixed> $configuration
 * @property array<string, mixed> $state
 * @property array<string, mixed> $snapshot
 * @property-read Run $run,
 *
 * @method static MapFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
class RunData extends Model
{
    protected $fillable = [
        'run_id',
    ];

    protected $casts = [
        'state' => AsArrayObject::class,
        'configuration' => AsArrayObject::class,
        'snapshot' => AsArrayObject::class,
    ];

    protected $attributes = [
        'state' => '{}',
        'configuration' => '{}',
        'snapshot' => '{}',
    ];

    public function getTable()
    {
        return config('game-engine.runs.data.table', parent::getTable());
    }

    /**
     * @return BelongsTo<Run, RunData>
     */
    public function run(): BelongsTo
    {
        return $this->belongsTo(game_engine()->runClass());
    }

    public function reset_state(): void
    {
        $this->state = [];
    }

    public function set_state(string $key, mixed $value): void
    {
        data_set($this->state, $key, $value);
    }

    public function forget_state(string $key): void
    {
        Arr::forget($this->state, $key);
    }

    public function get_state(string $key, mixed $default = null): mixed
    {
        return data_get($this->state, $key, $default);
    }

    public function append_state(string $key, mixed $value): void
    {
        $state = $this->get_state($key, []);

        if (!is_array($state)) {
            throw RunnerException::cannot_append_state($key);
        }

        $state[] = $value;

        $this->set_state($key, $state);
    }

    public function is_paused(): PauseInfo|false
    {
        $pause_info = data_get($this->state, 'paused');

        if ($pause_info === null) {
            return false;
        }

        return PauseInfo::from_array($pause_info);
    }

    public function resume(): self
    {
        $this->state['paused'] = null;

        return $this;
    }

    public function pause(PlayableActivity $activity): self
    {
        $this->state['paused'] = PauseInfo::from_activity($activity)->to_array();

        return $this;
    }

    public function awards(array $filter_award_ids = null): Collection
    {
        $story = collect($this->state['stories'] ?? [])->first() ?? [];
        $awards = data_get($story, 'result.awards', []);

        if ($filter_award_ids === null) {
            return collect($awards);
        }

        return collect($awards)->only($filter_award_ids);
    }
}
