<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Policies;

use DefStudio\GameEngine\Models\Run;
use DefStudio\GameEngine\Models\User;
use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\Permission;
use Illuminate\Auth\Access\HandlesAuthorization;

class RunPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        return $user->can(Permission::view_runs->value);
    }

    public function view(User $user, Run $target): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        if ($target->story->organization_id !== $user->current_organization_id) {
            return false;
        }

        if ($user->can(Permission::view_organizations->value) && $user->can(Permission::view_runs->value)) {
            return true;
        }

        if (!$user->can(Permission::view_runs->value)) {
            return false;
        }

        if (Feature::team_stories->enabled()) {
            return $target->user->belongs_to_any_team($user->teams);
        }

        return true;
    }

    public function play(User $user, Run $target): bool
    {
        if (!app(StoryPolicy::class)->play($user, $target->story)) {
            return false;
        }

        return $target->user_id === $user->id;
    }

    public function update(User $user, Run $target): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        if ($target->story->organization_id !== $user->current_organization_id) {
            return false;
        }

        if ($user->can(Permission::view_organizations->value) && $user->can(Permission::update_runs->value)) {
            return true;
        }

        if (Feature::team_stories->enabled()) {
            $target->loadMissing('user');

            if ($user->can(Permission::view_teams->value) && !$target->user->belongs_to_any_team($user->teams)) {
                return false;
            }
        }

        return $user->can(Permission::update_runs->value);
    }

    public function delete(User $user, Run $target): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        if ($target->story->organization_id !== $user->current_organization_id) {
            return false;
        }

        if ($user->can(Permission::view_organizations->value) && $user->can(Permission::delete_runs->value)) {
            return true;
        }

        if (Feature::team_stories->enabled()) {
            $target->loadMissing('user');

            if ($user->can(Permission::view_teams->value) && !$target->user->belongs_to_any_team($user->teams)) {
                return false;
            }
        }

        return $user->can(Permission::delete_runs->value);
    }
}
