<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use Illuminate\Http\Response;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

trait RunsStorytellings
{
    /** @var array */
    public $current_storytellings = [];

    protected function get_storytelling_scope_string(string|PlayableActivity $scope): string
    {
        if (is_string($scope)) {
            return $scope;
        }

        return str($scope::class)->classBasename()->remove('Playable')->lower()->toString();
    }

    public function complete_storytelling(PlayableActivity|string $activity, StorytellingType|string $type, bool $inline = false): void
    {
        if (is_string($activity)) {

            $activity = rescue(fn() => $this->{$activity});

            if ($activity === null) {
                return;
            }

            abort_unless($activity instanceof PlayableActivity, Response::HTTP_FORBIDDEN);
        }

        if (is_string($type)) {
            $type = StorytellingType::from($type);
        }

        $activity->complete_storytelling($type);

        $this->call_traits_method('complete_storytelling', $activity, $type);

        if ($this->has_inline_storytellings() && $inline) {
            $this->current_storytellings = [];
        } elseif (!$this->has_inline_storytellings() && !$inline) {
            /** @phpstan-ignore-next-line */
            $this->close_modal();
        }
    }

    protected function has_inline_storytellings(): bool
    {
        if (!property_exists($this, 'inline_storytellings')) {
            return false;
        }

        return $this->inline_storytellings;
    }

    protected function show_storytelling(PlayableActivity $activity, StorytellingType ...$type): bool
    {
        $type = collect($type)->first(fn(StorytellingType $t) => $activity->has_pending_storytellings($t));

        if ($type === null) {
            return false;
        }

        /** @noinspection PhpUndefinedFieldInspection */
        /** @phpstan-ignore-next-line  */
        $storytellings = $activity->storytellings;

        $storytellings_data = [
            'scope' => $this->get_storytelling_scope_string($activity),
            'type' => $type->name,
            'storytelling_ids' => $storytellings->where('type', $type)->pluck('id')->toArray(),
        ];

        if ($this->has_inline_storytellings()) {
            $this->current_storytellings = $storytellings_data;
        } else {
            /** @phpstan-ignore-next-line */
            $this->open_modal('game-engine.runner.storytelling', [...$storytellings_data, 'isModal' => true]);
        }

        return true;
    }
}
