<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnused */

/** @noinspection LaravelUnknownEloquentFactoryInspection */

namespace DefStudio\GameEngine\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\AsArrayObject;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use DefStudio\GameEngine\Database\Factories\MapFactory;
use DefStudio\GameEngine\Database\Factories\RunFactory;
use function game_engine;

/**
 * @property int $id;
 * @property int $story_id;
 * @property int $user_id;
 * @property Carbon|null $started_at;
 * @property Carbon|null $completed_at;
 * @property Carbon|null $available_from_date;
 * @property Carbon|null $available_until_date;
 * @property array<string, mixed> $state
 * @property array<string, mixed> $snapshot
 * @property-read Story $story,
 * @property-read User $user,
 *
 * @method static MapFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
class Run extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'story_id',
    ];

    protected $casts = [
        'state' => AsArrayObject::class,
        'configuration' => AsArrayObject::class,
        'snapshot' => AsArrayObject::class,
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'available_from_date' => 'datetime',
        'available_until_date' => 'datetime',
    ];

    protected $attributes = [
        'state' => '{}',
        'configuration' => '{}',
        'snapshot' => '{}',
    ];

    public function getTable()
    {
        return config('game-engine.runs.table', parent::getTable());
    }

    protected static function newFactory(): RunFactory
    {
        return RunFactory::new();
    }

    public function scopeOfStory(Builder $query, Story|int $story): void
    {
        $story_id = $story instanceof Story ? $story->id : $story;

        $query->where('story_id', $story_id);
    }

    public function scopeForUser(Builder $query, User|int $user): void
    {
        $user_id = $user instanceof User ? $user->id : $user;

        $query->where('user_id', $user_id);
    }

    public function scopeCompleted(Builder $query): void
    {
        $query->whereNotNull('completed_at');
    }

    public function scopeOngoing(Builder $query): void
    {
        $query->whereNull('completed_at');
    }

    /**
     * @return BelongsTo<Story, Run>
     */
    public function story(): BelongsTo
    {
        return $this->belongsTo(game_engine()->storyClass());
    }

    /**
     * @return BelongsTo<User, Run>
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(game_engine()->userClass());
    }

    public function set_state(string $key, mixed $value): void
    {
        data_set($this->state, $key, $value);
    }

    public function get_state(string $key, mixed $default = null): mixed
    {
        return data_get($this->state, $key, $default);
    }
}
