<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */ /** @noinspection PhpDocSignatureIsNotCompleteInspection */
/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Actions\Runs;

use Carbon\Carbon;
use Carbon\CarbonInterval;
use DefStudio\Actions\Action;

/**
 * @method static CarbonInterval run(array $timers);
 */
class ComputeActivityTime extends Action
{
    public function handle(array $timers): CarbonInterval
    {
        $duration = CarbonInterval::seconds(0);

        if (!isset($timers['current'])) {
            return $duration;
        }

        $duration = $duration->addSeconds($this->compute_frame_duration($timers['current'])->seconds);

        return collect($timers['backups'] ?? [])
            ->reduce(fn(CarbonInterval $duration, array $frame) => $duration->addSeconds($this->compute_frame_duration($frame)->seconds), $duration);
    }

    private function compute_frame_duration(array $frame): CarbonInterval
    {
        /** @var array{previous: Carbon, duration: CarbonInterval} $data */
        $data = collect([
            ...($frame['ticks'] ?? []),
            $frame['end'] ?? null,
        ])->filter()
            ->map(fn(string $tick) => carbon($tick))
            ->reduce(function(array $data, Carbon $tick): array {
                $previous = $data['previous'];
                $duration = $data['duration'];

                $measured_seconds = $tick->diffInSeconds($previous);

                $max_tick_duration = config('game-engine.runs.timers.max_tick_duration_seconds', 0);

                if ($max_tick_duration === 0 || $measured_seconds <= $max_tick_duration) {
                    $duration->addSeconds($measured_seconds);
                }

                return [
                    'duration' => $duration->addSeconds(),
                    'previous' => $tick,
                ];
            }, [
                'duration' => CarbonInterval::seconds(0),
                'previous' => carbon($frame['start']),
            ]);

        return $data['duration'];
    }
}
