<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

/** @noinspection PhpUnused */

namespace DefStudio\GameEngine\Decorators\Tasks;

use Illuminate\Support\Str;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\Http\Livewire\Tasks\Edit;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use DefStudio\GameEngine\Actions\Tasks\StoreAssetImage;
use DefStudio\GameEngine\Attributes\CallableFromLivewire;

class DragAndDrop extends TaskDecorator
{
    protected array $media_collections = [
        MediaCollection::base_image,
        MediaCollection::tile_placeholders,
    ];

    public static function type_label(): string
    {
        return __('Drag And Drop');
    }

    public static function configuration_component(): string
    {
        return 'game-engine::tasks.drag-and-drop.configuration';
    }

    public static function run_component(): string
    {
        return 'game-engine::tasks.drag-and-drop.run';
    }

    public function saving_from_livewire_editor(Edit $edit): void
    {
        if (!empty($edit->image_assets[MediaCollection::base_image->name])) {
            StoreAssetImage::run($this, $edit->image_assets[MediaCollection::base_image->name], MediaCollection::base_image);
            $edit->image_assets[MediaCollection::base_image->name] = null;
        }

        foreach (array_keys($this->tiles()) as $tile_index) {

            if (empty($edit->image_assets['tile_'.$tile_index])) {
                continue;
            }

            $media = StoreAssetImage::run($this, $edit->image_assets['tile_'.$tile_index], MediaCollection::tile_placeholders);

            $this->configuration['tiles'][$tile_index]['media_id'] = $media->id;

            $edit->image_assets['tile_'.$tile_index] = null;
        }

        $this->getMedia(MediaCollection::tile_placeholders)
            ->reject(fn(Media $media) => collect($this->tiles())->pluck('media_id')->contains($media->id))
            ->each(fn(Media $media) => $media->delete());

        $this->unsetRelation('media');
    }

    public function tiles(): array
    {
        return $this->configuration['tiles'] ?? [];
    }

    public function should_randomize_tiles_order(): bool
    {
        return $this->configuration['randomize_tiles_order'] ?? false;
    }

    public function is_repositioning_allowed(): bool
    {
        return $this->configuration['repositioning_allowed'] ?? false;
    }

    public function is_partial_submit_allowed(): bool
    {
        return $this->configuration['partial_submit_allowed'] ?? false;
    }

    public function has_loose_snaps(): bool
    {
        return $this->configuration['loose_snaps'] ?? false;
    }

    public function snaps(): array
    {
        return collect($this->tiles())
            ->reject(fn(array $tile_data) => $tile_data['should_not_be_placed'] ?? false)
            ->map(fn(array $tile_data): array => [...$tile_data['anchor'], 'distance' => $tile_data['snap'] ?? 10])
            ->values()
            ->toArray();
    }

    /**
     * @return array{repositioning_allowed: bool, partial_submit_allowed: bool, loose_snaps: bool}
     */
    public function run_options(): array
    {
        return [
            'repositioning_allowed' => $this->is_repositioning_allowed(),
            'partial_submit_allowed' => $this->is_partial_submit_allowed(),
            'loose_snaps' => $this->has_loose_snaps(),
        ];
    }

    public function question(): array
    {
        return $this->configuration['question'] ?? [];
    }

    #[CallableFromLivewire]
    public function remove_base_image(Edit $component): self
    {
        $this->getFirstMedia(MediaCollection::base_image)?->delete();
        $this->unsetRelation('media');

        $component->image_assets['base_image'] = null;

        return $this;
    }

    #[CallableFromLivewire]
    public function add_tile(): self
    {
        $this->configuration['tiles'][] = [
            'id' => Str::uuid(),
            'media_id' => null,
            'anchor' => ['x' => 0, 'y' => 0],
            'snap' => 50,
        ];

        return $this;
    }

    /** @noinspection PhpUnusedParameterInspection */
    #[CallableFromLivewire]
    public function delete_tile(Edit $component, int $index): self
    {
        unset($this->configuration['tiles'][$index]);

        return $this;
    }

    public function get_tile(string $id): array
    {
        $tile = collect($this->tiles())->first(fn(array $tile) => $tile['id'] === $id);

        throw_if($tile === null, 'Invalid tile ID');

        return $tile;
    }

    /** @noinspection PhpUnusedParameterInspection */
    #[CallableFromLivewire]
    public function remove_tile_placeholder(Edit $component, int $tile_index): self
    {
        $media_id = $this->tiles()[$tile_index]['media_id'];

        if (empty($media_id)) {
            return $this;
        }

        $media = $this->media->find($media_id);

        $media?->delete();

        $this->configuration['tiles'][$tile_index]['media_id'] = null;

        $this->save();

        $this->unsetRelation('media');

        return $this;
    }

    /** @noinspection PhpUnusedParameterInspection */
    #[CallableFromLivewire]
    public function move_tile_placeholder(Edit $component, int $index, int $offset_x, int $offset_y): self
    {
        $this->configuration['tiles'][$index]['anchor']['x'] += $offset_x;
        $this->configuration['tiles'][$index]['anchor']['y'] += $offset_y;

        return $this;
    }

    public function is_complete(array $state): bool
    {
        return isset($state['total_placements']);
    }

    public function process_completion(array $state): array
    {
        /** @var array<string, array> $total_placements */
        $total_placements = data_get($state, 'result.placements', []);

        $correct_placements = collect($total_placements)
            ->filter(fn(array $placement, string $tile_id) => $this->is_tile_in_right_position($tile_id, $placement));

        data_set($state, 'result.needed_placements', count($this->snaps()));
        data_set($state, 'result.total_placements', count($total_placements));
        data_set($state, 'result.wrong_placements', count($total_placements) - count($correct_placements));
        data_set($state, 'result.correct_placements', count($correct_placements));
        data_set($state, 'result.correct_placements_percent', round(count($correct_placements) / max(count($this->snaps()), count($total_placements)), 2));

        return $state;
    }

    public function is_tile_in_right_position(string $tile_id, array $placement): bool
    {
        $tile = $this->get_tile($tile_id);

        if ($tile['should_not_be_placed'] ?? false) {
            return false;
        }

        return $tile['anchor']['x'] === $placement['x'] && $tile['anchor']['y'] === $placement['y'];
    }

    public function get_awards_for_run(array $state): array
    {
        $awards = [];

        collect(data_get($state, 'result.placements', []))
            ->filter(fn(array $placement, string $tile_id) => $this->is_tile_in_right_position($tile_id, $placement))
            ->each(function(array $placement, string $tile_id) use (&$awards): void {
                $tile = $this->get_tile($tile_id);
                $this->assign_awards($awards, $tile['awards'] ?? []);
            });

        collect(data_get($state, 'result.placements', []))
            ->filter(fn(array $placement, string $tile_id) => !$this->is_tile_in_right_position($tile_id, $placement))
            ->each(function(array $placement, string $tile_id) use (&$awards): void {
                $tile = $this->get_tile($tile_id);
                $this->assign_awards($awards, $tile['failure_awards'] ?? []);
            });

        if ($state['result']['total_placements'] === $state['result']['correct_placements']) {
            $this->assign_awards($awards, $this->configuration['completion_awards'] ?? []);
        } else {
            $this->assign_awards($awards, $this->configuration['failure_awards'] ?? []);
        }

        return $awards;
    }

    private function assign_awards(array &$assigned_awards, array $awards): void
    {
        foreach ($awards as $award_id => $award_quantity) {
            if (!isset($assigned_awards[$award_id])) {
                $assigned_awards[$award_id] = 0;
            }

            $assigned_awards[$award_id] += (float) $award_quantity;
        }
    }
}
