<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\TimeUnit;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Exceptions\RunnerException;
use DefStudio\GameEngine\Decorators\Runs\PlayableMap;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

/**
 * @property-read PlayableMap $map
 */
trait RunsMaps
{
    public function getMapProperty(): PlayableMap
    {
        throw_if(!$this->is_map_active(), RunnerException::no_active_map());

        $map = $this->run->story->maps->sortBy('order')->values()->get($this->current_map_index());

        return PlayableMap::build($map)->setup($this->run);
    }

    protected function current_map_index(): int
    {
        return $this->run->get_state('current_map', -1);
    }

    protected function is_map_active(): bool
    {
        return $this->current_map_index() >= 0;
    }

    protected function start_runs_maps(): void
    {
        if (!$this->is_map_active()) {
            return;
        }

        $this->map->start_timer();
        $this->save();

        if ($this->show_storytelling($this->map, StorytellingType::activation, StorytellingType::prologue)) {
            return;
        }

        if ($this->is_mission_active()) {
            return;
        }

        $this->next_mission();
    }

    protected function next_map(bool $force = false): void
    {

        if ($this->is_map_active()) {
            throw_unless($force || $this->map->completed(), RunnerException::incomplete_map($this->map));

            if ($this->current_map_index() >= $this->run->story->maps->count() - 1) {
                $this->complete_story();

                return;
            }
        }

        $this->activate_map($this->current_map_index() + 1);

        $this->map->setup_storytelling();

        $this->reset_mission_state();

        $this->map->start_timer();
        $this->save();

        $this->reload_page();
    }

    protected function activate_map(int $index): void
    {
        $this->run->set_state('current_map', $index);
        $this->forgetComputed('map');
        $this->emit('map:activated');
    }

    protected function complete_storytelling_runs_maps(PlayableActivity $activity, StorytellingType $type): void
    {
        if (!($activity instanceof PlayableMap)) {
            return;
        }

        if ($type === StorytellingType::activation && !$this->show_storytelling($activity, StorytellingType::prologue)) {
            $this->next_mission();

            return;
        }

        if ($type === StorytellingType::prologue) {
            $this->next_mission();

            return;
        }

        if ($type === StorytellingType::epilogue) {
            $this->complete_map();
        }

        if ($type === StorytellingType::timeout) {
            $this->force_map_timeout();
        }
    }

    protected function complete_map(StorytellingType $storytelling_type = StorytellingType::epilogue, bool $force = false): void
    {
        if ($this->show_storytelling($this->map, $storytelling_type)) {
            return;
        }

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->assign_awards($this->map);
        }

        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->assign_level($this->map);
        }

        $this->map->stop_timer();
        $this->save();

        $this->next_map($force);
    }

    public function get_map_deadline(): ?string
    {
        if (empty($this->map->configuration['time_limit'])) {
            return null;
        }

        $elapsed_seconds = $this->map->get_elapsed_time()->totalSeconds;

        $remaining_seconds = TimeUnit::from($this->map->configuration['time_limit_unit'])->to_seconds($this->map->configuration['time_limit']) - $elapsed_seconds;

        return now()->addSeconds($remaining_seconds)->format('Y-m-d H:i:s');
    }

    public function force_map_timeout(): void
    {
        if (empty($this->map->configuration['time_limit'])) {
            return;
        }

        $this->complete_map(StorytellingType::timeout, true);
    }
}
