<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Models\Concerns;

use DefStudio\GameEngine\Models\Story;
use DefStudio\GameEngine\Exceptions\StoryException;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use DefStudio\GameEngine\Events\Stories\SubscribedToStory;
use DefStudio\GameEngine\Events\Stories\UnsubscribedFromStory;
use DefStudio\GameEngine\Models\Contracts\SubscribesToStories;

trait HasStories
{
    public static function bootHasStories(): void
    {
        self::deleting(fn(SubscribesToStories $model) => $model->stories()->detach());
    }

    /**
     * @return MorphToMany<Story>
     */
    public function stories(): MorphToMany
    {
        return $this->morphToMany(game_engine()->storyClass(), 'subscriber', config('game-engine.stories.subscriptions_table'));
    }

    public function is_subscribed_to_story(Story $story): bool
    {
        return $this->has_attached($story);
    }

    public function subscribe_to_story(Story $story): self
    {
        /** @phpstan-ignore-next-line */
        $organization_id = method_exists($this, 'current_organization') ? $this->current_organization_id : $this->organization_id;

        throw_if($organization_id !== $story->organization_id, StoryException::organization_membership_failure());
        throw_if($this->is_subscribed_to_story($story), StoryException::already_subscribed());

        $this->stories()->attach($story->id);

        SubscribedToStory::dispatch($this, $story);

        return $this;
    }

    public function unsubscribe_from_story(Story $story): self
    {
        throw_if(!$this->is_subscribed_to_story($story), StoryException::subscription_failure());

        $this->stories()->detach($story);

        UnsubscribedFromStory::dispatch($this, $story);

        return $this;
    }
}
