<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Exceptions\RunnerException;
use DefStudio\GameEngine\Decorators\Runs\PlayableTask;

/**
 * @property-read PlayableTask $task
 */
trait RunsTasks
{
    protected function is_task_active(): bool
    {
        return $this->current_task_index() >= 0;
    }

    protected function current_task_index(): int
    {
        return $this->run->get_state('current_task', -1);
    }

    public function getTaskProperty(): PlayableTask
    {
        return $this->mission->playable_tasks()
            ->values()
            ->get($this->current_task_index()) ?? throw RunnerException::no_active_task();
    }

    protected function next_task(): void
    {
        if ($this->is_task_active()) {
            throw_unless($this->task->completed(), RunnerException::incomplete_task($this->task));

            if ($this->current_task_index() >= $this->mission->playable_tasks()->count() - 1) {
                $this->complete_mission();

                return;
            }
        }

        $this->activate_task($this->current_task_index() + 1);

        $this->task->start_timer();
        $this->run->save();
    }

    protected function activate_task(int $index): void
    {
        $this->reset_current_state();

        $this->run->set_state('current_task', $index);

        $this->forgetComputed('task');

        $this->task->setup_storytelling();

        $this->show_storytelling($this->task, StorytellingType::activation);

        $this->run->save();


        $this->emit('task:activated');
    }

    protected function reset_current_state(): void
    {
        $this->run->set_state('current_task', -1);
        $this->forgetComputed('task');
    }

    public function complete_task_with_result(array $result): void
    {
        $state = $this->task->get_state();
        $state['result'] = $result;
        $state = $this->task->process_completion($state);

        $this->task->replace_state($state);

        $this->run->save();

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->assign_awards($this->task);
        }

        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->assign_level($this->task);
        }

        $this->task->stop_timer();
        $this->run->save();

        $this->next_task();
    }
}
