<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpDocSignatureIsNotCompleteInspection */

namespace DefStudio\GameEngine\Actions\Awards;

use DefStudio\Actions\Action;
use DefStudio\GameEngine\Models\Award;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Collection;
use DefStudio\GameEngine\Decorators\ModelDecorator;
use DefStudio\GameEngine\Models\Contracts\FitsInScope;
use DefStudio\GameEngine\Models\Contracts\AwardsDefiner;

/**
 * @method static Collection<int, Award> run(ModelDecorator|Model|null $model)
 */
class GetAwardsForScope extends Action
{
    /**
     * @return Collection<int, Award>
     */
    public function handle(ModelDecorator|Model|null $model): Collection
    {
        if ($model instanceof ModelDecorator) {
            $model = $model->model();
        }

        if ($model === null) {
            return Collection::empty();
        }

        $awards = Collection::empty();

        if ($model instanceof AwardsDefiner) {
            $awards->push(...$model->awards->whereNull('parent_id')->sortByDesc('order'));
        }

        if ($model instanceof FitsInScope) {
            $model->traverse_parent_scopes(function(Model $parent) use ($awards): void {
                if ($parent instanceof AwardsDefiner) {
                    $awards->push(...$parent->awards->whereNull('parent_id')->sortByDesc('order'));
                }
            });
        }

        $awards->push(...game_engine()->awardQuery()->whereNull('scope_id')->whereNull('parent_id')->orderBy('order', 'desc')->get());

        /** @var Collection<int, Award> $awards */
        $awards = Collection::make([...$awards->reverse()->values()]);

        return $awards;
    }
}
