<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are prohibited without the prior written permission of DEF STUDIO S.R.L. This software is provided "as is" and any express or implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
 */

namespace DefStudio\GameEngine\Http\Livewire\Organizations;

use Livewire\Component;
use Illuminate\Contracts\View\View;
use DefStudio\GameEngine\Enums\Role;
use Illuminate\Support\Facades\Gate;
use DefStudio\GameEngine\Models\User;
use DefStudio\GameEngine\Models\Organization;
use DefStudio\GameEngine\Actions\Organizations\SaveOrganization;

class Edit extends Component
{
    /** @var Organization */
    public $organization;

    public ?string $role_under_creation = null;

    protected $listeners = [
        'user.created' => 'handle_user_created',
        'user.switch' => 'handle_user_switch',
    ];

    public function mount(): void
    {
        Gate::authorize('save', $this->organization);
    }

    public function rules(): array
    {
        return [
            'organization.name' => ['required', 'string', 'max:255'],
        ];
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.organizations.edit');
    }

    public function save(): void
    {
        $this->validate();

        SaveOrganization::run($this->organization);

        $this->redirect(route('game-engine.admin.organizations.edit', $this->organization));
    }

    public function handle_user_created(int $user_id): void
    {
        /** @var User $user */
        $user = game_engine()->userQuery()->find($user_id);

        $role = Role::from($this->role_under_creation);

        $user->subscribe_to_organization($this->organization, $role, $user->organizations()->count() === 0);
        $this->emit('user.subscribed', $user->id, $this->organization->id);
    }

    public function handle_user_switch(int $user_id): void
    {
        /** @var User $user */
        $user = game_engine()->userQuery()->find($user_id);

        $role = Role::from($this->role_under_creation);

        $user->subscribe_to_organization($this->organization, $role, $user->organizations()->count() === 0);
        $this->emit('user.subscribed', $user->id, $this->organization->id);
    }
}
