<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Http\Livewire\Runs;

use Illuminate\Contracts\View\View;
use DefStudio\GameEngine\Models\Run;
use Illuminate\Support\Facades\Gate;
use LivewireUI\Modal\ModalComponent;
use DefStudio\GameEngine\Enums\TimeUnit;
use DefStudio\GameEngine\Models\Mission;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Decorators\Runs\PlayableMission;
use DefStudio\GameEngine\Http\Livewire\Concerns\ImprovedModal;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\RunsTasks;
use DefStudio\GameEngine\Http\Livewire\Concerns\CallsTraitsMethods;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\RunsStorytellings;

/**
 * @property-read Run $run;
 * @property-read PlayableMission $mission
 */
class MissionModal extends ModalComponent
{
    use CallsTraitsMethods;
    use ImprovedModal;
    use RunsStorytellings;
    use RunsTasks;

    public int $run_id;

    public int $mission_id;

    protected bool $inline_storytellings = true;

    protected $listeners = [
        'complete_storytelling',
    ];

    public static function modalMaxWidth(): string
    {
        return config('game-engine.missions.modal_width', '4xl');
    }

    public function mount(): void
    {
        Gate::authorize('play', $this->run);
    }

    public function booted(): void
    {
        if ($this->is_task_active()) {
            $this->task->tick_timer();
            $this->run->save();
        }
    }

    public function getRunProperty(): Run
    {
        return Run::findOrFail($this->run_id);
    }

    public function getMissionProperty(): PlayableMission
    {
        return PlayableMission::build(Mission::findOrFail($this->mission_id))->setup($this->run);
    }

    public function render(): View
    {
        return view('game-engine::livewire.runs.mission-modal');
    }

    public function start(): void
    {
        if ($this->is_task_active()) {
            $this->task->start_timer();
            $this->run->save();

            return;
        }

        $this->next_task();

    }

    protected function complete_mission(StorytellingType $storytelling_type = StorytellingType::epilogue, bool $force = false): void
    {
        $this->emit(
            'complete_mission', $storytelling_type->value, $force
        );
    }

    public function get_mission_deadline(): ?string
    {
        if (empty($this->mission->configuration['time_limit'])) {
            return null;
        }

        $elapsed_seconds = $this->mission->get_elapsed_time()->totalSeconds;

        $remaining_seconds = TimeUnit::from($this->mission->configuration['time_limit_unit'])->to_seconds($this->mission->configuration['time_limit']) - $elapsed_seconds;

        return now()->addSeconds($remaining_seconds)->format('Y-m-d H:i:s');
    }

    public function force_mission_timeout(): void
    {
        if (empty($this->mission->configuration['time_limit'])) {
            return;
        }

        $this->complete_mission(StorytellingType::timeout, true);
    }
}
