<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Http\Livewire\Maps;

use DefStudio\GameEngine\Enums\TimeUnit;
use Livewire\Component;
use Illuminate\Contracts\View\View;
use DefStudio\GameEngine\Models\Map;
use DefStudio\GameEngine\Actions\Maps\SaveMap;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\DTO\Missions\Placeholder;
use DefStudio\GameEngine\Actions\Missions\CreateMission;
use DefStudio\GameEngine\Actions\Missions\DeleteMission;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use DefStudio\GameEngine\Actions\Maps\GetMissionPlaceholders;
use DefStudio\GameEngine\Actions\Livewire\Sortable\UpdateOrder;
use DefStudio\GameEngine\Actions\Missions\UpdateMissionFromPlaceholder;

class Editor extends Component
{
    use AuthorizesRequests;

    protected $listeners = [
        'mission.updated' => 'refresh_mission_placeholders',
        'map.saved' => '$refresh',
    ];

    /** @var Map */
    public $map;

    /** @var array<int, Placeholder> */
    public array $mission_placeholders = [];

    /** @var int */
    public $time_limit;

    public string $time_limit_unit;

    public bool $dirty = false;

    public function mount(): void
    {
        $this->authorize('save', $this->map);

        $this->time_limit = $this->map->configuration['time_limit'] ?? '';
        $this->time_limit_unit = $this->map->configuration['time_limit_unit'] ?? TimeUnit::minutes->value;

        $this->refresh_mission_placeholders();
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.maps.editor');
    }

    public function updated(string $key): void
    {
        if ($key === 'mission_placeholders') {
            return;
        }

        $this->dirty = true;
    }

    public function getBackgroundImageUrlProperty(): string
    {
        return $this->map->getFirstMediaUrl(MediaCollection::background);
    }

    public function add_mission(): void
    {
        $mission = CreateMission::run($this->map);

        $this->refresh_mission_placeholders();

        $this->emit('openModal', 'game-engine.admin.missions.edit', ['isModal' => true, 'mission_id' => $mission->id]);
    }

    public function delete_mission(int $mission_id): void
    {
        $mission = $this->map->missions()->findOrFail($mission_id);
        DeleteMission::run($mission);

        $this->refresh_mission_placeholders();
    }

    public function updatedMissionPlaceholders($new_data): void
    {
        collect($new_data)
            ->mapInto(Placeholder::class)
            ->filter(function(Placeholder $placeholder): bool {
                $mission = $this->map->missions()->with('map')->find($placeholder->id);

                return UpdateMissionFromPlaceholder::run($mission, $placeholder);
            })->each(fn(Placeholder $placeholder) => $this->emit('mission.updated', $placeholder->id));

        $this->refresh_mission_placeholders();
    }

    public function refresh_mission_placeholders(): void
    {
        $this->map->unsetRelation('missions');
        $this->mission_placeholders = GetMissionPlaceholders::run($this->map)->toArray();
    }

    public function update_missions_order(array $new_order): void
    {
        UpdateOrder::run($this->map->missions, $new_order);
    }

    public function save(): void
    {
        $this->map->configuration['time_limit'] = empty($this->time_limit) ? null : $this->time_limit;
        $this->map->configuration['time_limit_unit'] = empty($this->time_limit_unit) ? null : $this->time_limit_unit;

        SaveMap::run($this->map);

        $this->dirty = false;
    }
}
