<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Http\Livewire\Users;

use Illuminate\Validation\Rule;
use Illuminate\Contracts\View\View;
use DefStudio\GameEngine\Enums\Role;
use LivewireUI\Modal\ModalComponent;
use DefStudio\GameEngine\Models\User;
use DefStudio\GameEngine\Enums\Feature;
use Illuminate\Validation\Rules\Password;
use DefStudio\GameEngine\Actions\Users\SaveUser;
use DefStudio\GameEngine\Actions\Users\FindAvailableLocales;
use DefStudio\GameEngine\Http\Livewire\Concerns\ImprovedModal;
use DefStudio\GameEngine\Http\Livewire\Concerns\RedirectsBack;

class Edit extends ModalComponent
{
    use ImprovedModal;
    use RedirectsBack;

    /** @var User */
    public $user;

    public string $password = '';

    public array $available_locales;

    public string $creating_role = '';

    public function mount(): void
    {
        /** @phpstan-ignore-next-line */
        $this->user ??= game_engine()->userQuery()->make();

        $this->available_locales = FindAvailableLocales::run();

        SaveUser::make()->authorize($this->user, Role::tryFrom($this->creating_role));
    }

    public function rules(): array
    {
        $rules = [
            'user.name' => ['required', 'string', 'max:255'],
            'creating_role' => 'nullable',
        ];

        $role = empty($this->creating_role) ? $this->user->get_role_in_organization(game_engine()->user(true)?->current_organization)?->value : $this->creating_role;
        if ($role && config("game-engine.users.roles.$role.without_email", false)) {
            $rules['user.name'][] = Rule::unique('users', 'name')->ignore($this->user->id);
            $rules['password'] = [$this->user->exists ? 'nullable' : 'required', 'string', Password::min(6)->letters()->numbers()];
        } else {
            $rules['user.email'] = ['required', 'string', 'email', 'max:255', Rule::unique('users', 'email')->ignore($this->user->id)];
        }

        if (Feature::multi_language->enabled()) {
            $rules['user.locale'] = ['required', 'string', Rule::in(array_keys($this->available_locales))];
        }

        return $rules;
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.users.edit');
    }

    public function save(): void
    {
        $this->validate();

        SaveUser::run($this->user, Role::tryFrom($this->creating_role), $this->password);

        if ($this->isModal) {
            if ($this->user->wasRecentlyCreated) {
                $this->emit('user.created', $this->user->id);
            } else {
                $this->emit('user.updated', $this->user->id);
            }

            $this->closeModal();
        } else {
            $this->redirect_back();
        }
    }

    public function switch_to_user(int $user_id): void
    {
        if ($this->isModal) {
            $this->emit('user.switch', $user_id);
            $this->closeModal();
        } else {
            $this->reflash_back_url();
            $this->redirect(route('game-engine.admin.users.edit', $user_id));
        }
    }
}
