<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Policies;

use DefStudio\GameEngine\Models\User;
use DefStudio\GameEngine\Models\Story;
use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\Permission;
use Illuminate\Auth\Access\HandlesAuthorization;

class StoryPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        return $user->can(Permission::view_stories->value);
    }

    public function view(User $user, Story $target): bool
    {
        return $target->has_some_attached($user->current_teams);
    }

    public function play(User $user, Story $target): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        if ($target->organization_id !== $user->current_organization_id) {
            return false;
        }

        if (Feature::team_stories->enabled() && $target->has_some_attached($user->current_teams, 'teams')) {
            return true;

        }

        if (!Feature::user_stories->enabled()) {
            return false;
        }

        if ($user->id === game_engine()->user(true)?->id) {
            return $target->has_attached($user, 'users');
        }

        if (!game_engine()->user(true)?->can(Permission::create_runs->value)) {
            return $target->has_attached($user, 'users');
        }

        return true;
    }

    public function save(User $user, Story $target): bool
    {
        if ($target->exists) {
            return $this->update($user, $target);
        }

        return $this->create($user);
    }

    public function create(User $user): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        return $user->can(Permission::create_stories->value);
    }

    public function update(User $user, Story $target): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        if (!$user->can(Permission::edit_stories->value)) {
            return false;
        }

        return $target->organization_id === $user->current_organization_id;
    }

    public function delete(User $user, Story $target): bool
    {
        if ($user->current_organization_id === null) {
            return false;
        }

        if (!$user->can(Permission::delete_stories->value)) {
            return false;
        }

        return $target->organization_id === $user->current_organization_id;
    }
}
