<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are prohibited without the prior written permission of DEF STUDIO S.R.L. This software is provided "as is" and any express or implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Stories;

use DefStudio\GameEngine\Enums\Role;
use DefStudio\WiredTables\WiredTable;
use DefStudio\GameEngine\Models\Story;
use Illuminate\Database\Eloquent\Builder;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Database\Eloquent\Relations\Relation;
use DefStudio\GameEngine\Actions\Stories\DeleteStory;
use function game_engine;

class Table extends WiredTable
{
    protected function query(): Builder|Relation
    {
        /** @noinspection PhpVoidFunctionResultUsedInspection */
        $query = match (true) {
            $this->user_is_super_admin() => game_engine()->storyQuery(),
            $this->user_is_admin_with_active_organization() => game_engine()->user()->current_organization->stories(),
            default => abort(Response::HTTP_FORBIDDEN),
        };

        return $query
            ->withCount('maps')
            ->with('organization');
    }

    protected function columns(): void
    {
        if ($this->user_is_super_admin()) {
            $this->column(trans_choice('Organization|Organizations', 1), 'organization.name')->searchable()->sortable();
        }

        $this->column(__('Title'), 'title')->searchable()->sortable();
        $this->column(__('Description'), 'description')->wrapText(true);
        $this->column(trans_choice('Map|Maps', 2), 'maps_count');
        $this->column(trans_choice('Action|Actions', 2))->view('game-engine::livewire.admin.stories.table.actions');
    }

    public function delete(Story $story): void
    {
        DeleteStory::run($story);
    }

    private function user_is_admin_with_active_organization(): bool
    {
        if (!game_engine()->role(Role::admin)) {
            return false;
        }

        return (bool) game_engine()->user()->current_organization_id;
    }

    private function user_is_super_admin(): bool
    {
        return game_engine()->role(Role::super_admin);
    }
}
