<?php

/** @noinspection PhpUnhandledExceptionInspection */
/** @noinspection PhpUnused */

/** @noinspection PhpRedundantVariableDocTypeInspection */

/** @noinspection PhpPossiblePolymorphicInvocationInspection */

namespace DefStudio\GameEngine\Database\Seeders;

use Illuminate\Database\Seeder;
use DefStudio\GameEngine\Enums\Role;
use DefStudio\GameEngine\Models\Map;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use DefStudio\GameEngine\Models\Task;
use DefStudio\GameEngine\Models\User;
use DefStudio\GameEngine\Models\Story;
use Illuminate\Support\Facades\Artisan;
use DefStudio\GameEngine\Models\Mission;
use DefStudio\GameEngine\Models\Organization;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\Actions\Stories\DeleteStory;
use DefStudio\GameEngine\Decorators\Tasks\DragAndDrop;
use DefStudio\GameEngine\Decorators\Tasks\MultipleChoiceQuestion;

class DevelopmentSeeder extends Seeder
{
    public User $super_admin;

    public Organization $organization;

    public Story $story;

    public function run(): void
    {
        $this->setup();

        $this->create_super_admin();

        $this->create_test_organization();

        $this->create_story();

        $this->create_maps();
    }

    private function setup(): void
    {
        Artisan::call('game-engine:sync-roles');
    }

    private function create_super_admin(): void
    {
        if ($super_admin = game_engine()->userQuery()->role(Role::super_admin->name)->first()) {
            $this->super_admin = $super_admin;

            return;
        }

        /** @var User $super_admin */
        $this->super_admin = game_engine()->userQuery()->create([
            'name' => 'Admin',
            'email' => 'admin@defstudio.it',
            'password' => Hash::make('123456'),
        ]);

        $this->super_admin->assignRole(Role::super_admin->name);
    }

    private function create_test_organization(): void
    {

        $this->organization = game_engine()->organizationQuery()->firstOrCreate([
            'name' => 'def:studio',
        ]);

        $this->super_admin->switch_organization($this->organization);
        Auth::login($this->super_admin);
    }

    private function create_story(): void
    {
        if ($story = game_engine()->storyQuery()->where('title', 'test story')->first()) {
            DeleteStory::run($story);
        }

        /** @var class-string<Story> $story_class */
        $story_class = game_engine()->storyClass();

        $this->story = $story_class::factory()
            ->for($this->organization)
            ->with_prologue(
                text: '<p>Hey there, I’m Hal9000 the AI and chief mate of this ship, I will be guiding you in this adventure!</p>',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png',
            )
            ->with_prologue(
                text: '
                    <p>You are currently part of a crew that is on an expedition to find the legendary indigo stone!</p>
                    <p>On our home planet, we use these special minerals to power the indigo tower that creates portals between galaxies.</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->with_prologue(
                text: '
                    <p>Recently the all the stones in the tower suddenly disappeared due to overheating issues, so we need to find new ones in a secret abandoned space station located on a far distant planet.</p>
                    <p>If we don’t reactivate the indigo tower, all the other crews of our home planet that are out exploring other galaxies will never be able to come back home through the portals created by the tower.</p>
                    <p>Getting to the stones won’t be easy, we will have to complete different tests, but don’t worry, I will guide you through everything!</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->published()
            ->create(['title' => 'test story']);
    }

    private function create_maps(): void
    {
        $this->create_map_1();
        $this->create_map_2();
    }

    private function create_map_1(): void
    {
        /** @var class-string<Map> $map_class */
        $map_class = game_engine()->mapClass();

        /** @var Map $map */
        $map = $map_class::factory()
            ->for($this->story)
            ->with_background_image(__DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/background.jpg')
            ->with_prologue(
                text: '
                    <p>We’re almost ready to start our mission, but before we disembark you should take this brief test just to prepare yourself for this adventure!</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->with_epilogue(
                text: '
                    <p>Great job, you are clearly ready for the mission! Off we go then time to disembark!</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )->create(['title' => 'Station Entrance', 'description' => 'This is the external area of the structure; the children will find the entrance door locked with the entry door keypad turned off; the pad is connected to a circuit mechanism with a fuse box that is missing a battery pack. The children will have to find a battery pack located in the room to activate the circuit and turn on the keypad.']);

        $this->create_mission_1x1($map);

        $this->create_mission_1x2($map);

        $this->create_mission_1x3($map);
    }

    public function create_mission_1x1(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/battery_pack.png')
            ->with_activation_storytelling(
                text: '
                   <p>According to my data, this is the entrance of the space station, but all the doors seem to be connected to a special locking mechanism, I’m 99.933% sure that we can get in by using the keypad next to the door, but it doesn’t seem to be working, maybe we need to activate the circuit connected to it… if only we had a battery pack… </p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->with_epilogue(
                text: '<p>That’s a battery pack, just what we were looking for! The fuse box next to the circuit is working now, why don’t we check it out</p>',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Find the battery pack',
                'description' => 'the children will have to locate and click on the battery pack located in the room. Once the battery pack is selected, the fuse box connected to the circuit will turn on. The battery pack image disappears from the scene.',
                'order' => 1,
                'pos_x' => 43,
                'pos_y' => 994,
                'configuration' => [
                    'hide_when_completed' => true,
                ],
            ]);
    }

    public function create_mission_1x2(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission = $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/fuse_box.png')
            ->with_epilogue(
                text: '
                    <p>Puzzle completed! Great job, now that the circuit is working, we can access the keypad next to the door</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Fix the fuse box',
                'description' => 'now that the fuse box is active, the children will have to click on it, thus activating the first test of this room (TEST 1). Once the test is completed, the fuse box will release the electricity in the circuit; the keypad connected to the door turns on.',
                'order' => 2,
                'pos_x' => 1971,
                'pos_y' => 200,
                'configuration' => [
                    'hide_when_not_active' => true,
                ],
            ]);

        $this->create_tasks_1x2($mission);
    }

    public function create_tasks_1x2(Mission $mission): void
    {
        /** @var class-string<Task> $task_class */
        $task_class = game_engine()->taskClass();

        $task_class::factory()
            ->for($mission)
            ->create([
                'type' => MultipleChoiceQuestion::class,
                'configuration' => [
                    'question' => '<p>Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. <strong>Duis aute irure dolor in reprehenderit</strong> in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.</p>',
                    'answers' => [
                        ['id' => 'e953932a-b772-49b0-9353-cb6466f0a56d', 'answer' => '<p>aboris nisi ut</p>', 'correct' => false],
                        ['id' => '0d46804e-562f-4efa-8a17-740bee6b4e36', 'answer' => '<p>commodo <i>consequat</i>.</p>', 'correct' => true],
                        ['id' => 'd251838a-c5b2-44ba-97bf-5017c18fae5e', 'answer' => '<p>mollit anim</p>', 'correct' => false],
                        ['id' => '9ed78fa6-6459-47b9-88a4-8fd534314960', 'answer' => '<p>exercitation ullamco</p>', 'correct' => false],
                    ],
                    'randomize_answers_order' => true,
                ],
            ]);

        $task_class::factory()
            ->for($mission)
            ->create([
                'type' => MultipleChoiceQuestion::class,
                'configuration' => [
                    'question' => '<p>Sed ut perspiciatis unde omnis iste natus error sit voluptatem accusantium doloremque laudantium, totam rem aperiam, eaque ipsa quae ab illo inventore veritatis et quasi architecto beatae vitae dicta sunt explicabo. Nemo enim ipsam voluptatem quia voluptas sit aspernatur aut odit aut fugit, sed quia consequuntur magni dolores</p>',
                    'answers' => [
                        ['id' => 'e153932a-b772-49b0-9353-cb6466f0a56d', 'answer' => '<p>qui ratione voluptatem</p>', 'correct' => false],
                        ['id' => '0146804e-562f-4efa-8a17-740bee6b4e36', 'answer' => '<p>sed quia non numquam <i>consequat</i>.</p>', 'correct' => true],
                        ['id' => 'd151838a-c5b2-44ba-97bf-5017c18fae5e', 'answer' => '<p>minima veniam</p>', 'correct' => false],
                        ['id' => '91d78fa6-6459-47b9-88a4-8fd534314960', 'answer' => '<p>quis nostrum exercitationem</p>', 'correct' => false],
                    ],
                    'randomize_answers_order' => true,
                ],
            ]);

        $task_class::factory()
            ->for($mission)
            ->create([
                'type' => MultipleChoiceQuestion::class,
                'configuration' => [
                    'question' => '<p>At vero eos et accusamus et iusto odio dignissimos ducimus qui blanditiis praesentium voluptatum deleniti atque corrupti quos dolores et quas molestias excepturi sint occaecati cupiditate non provident, similique sunt in culpa qui officia deserunt mollitia animi, id est laborum et dolorum fuga. Et harum quidem rerum facilis est et expedita distinctio.</p>',
                    'answers' => [
                        ['id' => 'e253932a-b772-49b0-9353-cb6466f0a56d', 'answer' => '<p>qui ratione voluptatem</p>', 'correct' => false],
                        ['id' => '0246804e-562f-4efa-8a17-740bee6b4e36', 'answer' => '<p>sed quia non numquam <i>consequat</i>.</p>', 'correct' => true],
                        ['id' => 'd251838a-c5b2-44ba-97bf-5017c18fae5e', 'answer' => '<p>minima veniam</p>', 'correct' => false],
                        ['id' => '92d78fa6-6459-47b9-88a4-8fd534314960', 'answer' => '<p>quis nostrum exercitationem</p>', 'correct' => false],
                    ],
                    'randomize_answers_order' => true,
                ],
            ]);
    }

    public function create_mission_1x3(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission = $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/door_keypad.png')
            ->with_media(MediaCollection::placeholder_active, __DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/door_keypad_active.png')
            ->with_epilogue(
                text: '
                    <p>Great job, you managed to unlock the door and you also found an interesting item. Click on the icon on screen when you want to use it. This was great practice for all the real tests that you will find in the next rooms… Alright, on we go!</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Open the door',
                'description' => 'the children will have to click on the number keypad in order, this activated the second test of the room (TEST 2)',
                'order' => 3,
                'pos_x' => 1213,
                'pos_y' => 323,
            ]);

        $this->create_tasks_1x3($mission);
    }

    public function create_tasks_1x3(Mission $mission): void
    {
        /** @var class-string<Task> $task_class */
        $task_class = game_engine()->taskClass();

        /** @var Task $task */
        $task = $task_class::factory()
            ->for($mission)
            ->with_media(MediaCollection::base_image, __DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/task_1x3x1/base_image.png')
            ->with_media(MediaCollection::tile_placeholders, __DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/task_1x3x1/tile_1.png')
            ->with_media(MediaCollection::tile_placeholders, __DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/task_1x3x1/tile_2.png')
            ->with_media(MediaCollection::tile_placeholders, __DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/task_1x3x1/tile_3.png')
            ->with_media(MediaCollection::tile_placeholders, __DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/task_1x3x1/tile_4.png')
            ->with_media(MediaCollection::tile_placeholders, __DIR__.'/../../resources/fixtures/test_story/maps/0_tutorial/task_1x3x1/tile_5.png')
            ->create([
                'type' => DragAndDrop::class,
            ]);

        $media = $task->getMedia(MediaCollection::tile_placeholders)->values();

        $task->configuration = [
            'question' => '<p>At vero eos et accusamus et iusto odio dignissimos ducimus qui blanditiis praesentium voluptatum deleniti atque corrupti quos dolores et quas molestias excepturi sint occaecati cupiditate non provident, similique sunt in culpa qui officia deserunt mollitia animi, id est laborum et dolorum fuga. Et harum quidem rerum facilis est et expedita distinctio.</p>',
            'tiles' => [
                [
                    'id' => '76745248-2dcd-40b6-9249-d39ca76465cc',
                    'snap' => 10,
                    'anchor' => ['x' => 336, 'y' => 449],
                    'media_id' => $media->get(0)->id,
                ],
                [
                    'id' => 'f56b8481-bc98-4e0c-8f50-6b49b58c875f',
                    'snap' => 10,
                    'anchor' => ['x' => 223, 'y' => 225],
                    'media_id' => $media->get(1)->id,
                ],
                [
                    'id' => '821e1e6e-de0f-462b-8028-2185ba23208e',
                    'snap' => 10,
                    'anchor' => ['x' => 0, 'y' => 224],
                    'media_id' => $media->get(2)->id,
                ],
                [
                    'id' => '76244bfa-9b1d-4e0d-894e-193f8736ed83',
                    'snap' => 10,
                    'anchor' => ['x' => 223, 'y' => 0],
                    'media_id' => $media->get(3)->id,
                ],
                [
                    'id' => '708fc22b-0bc7-4268-ab11-ac661c4450cc',
                    'snap' => 10,
                    'anchor' => ['x' => 450, 'y' => 225],
                    'media_id' => $media->get(4)->id,
                ],
            ],
            'randomize_tiles_order' => true,
            'repositioning_allowed' => true,
            'partial_submit_allowed' => false,
            'loose_snaps' => true,
        ];

        $task->save();
    }

    private function create_map_2(): void
    {
        /** @var class-string<Map> $map_class */
        $map_class = game_engine()->mapClass();

        /** @var Map $map */
        $map = $map_class::factory()
            ->for($this->story)
            ->with_background_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/background.jpg')
            ->with_prologue(
                text: '
                    <p>Here we are in the very first room of the station, the keypad to access the next room seems to be turned off! Look, the pavement has some funny looking drawings…let’s have a closer look.</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->with_epilogue(
                text: '
                    <p>Eureka! First room cleared my friend, now let’s move on to the second one!</p>
                ',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )->create(['title' => 'Station Interior', 'description' => 'Upon entering the first room of the station, the children find a worn out tablet on the ground displaying the following message: “I was the captain of this station, I left due to my body growing weak, find the pieces to my name and you will earn what you seek.“ This message prompts the children to look for pieces of the name of the captain in all the rooms.']);

        $this->create_mission_2x1($map);
        $this->create_mission_2x2($map);
        $this->create_mission_2x3($map);
        $this->create_mission_2x4($map);
        $this->create_mission_2x5($map);
        $this->create_mission_2x6($map);
        $this->create_mission_2x7($map);
    }

    public function create_mission_2x1(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/puzzle.png')
            ->with_media(MediaCollection::placeholder_complete, __DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/puzzle_completed.png')
            ->with_epilogue(
                text: '<p>Great job! The drawings match the painting on the wall! I thought I saw something move near that planetarium… let’s check it out!</p>',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Solve the puzzle',
                'description' => 'The children click on the 4 tiles located on the pavement of the room; the first test of the room is triggered (TEST 1). Once the first test is completed, the tiles on the pavement will appear in the correct disposition, thus displaying the same drawings as the painting in the wall.',
                'order' => 1,
                'pos_x' => 1056,
                'pos_y' => 1217,
                'configuration' => [
                    'hide_when_completed' => false,
                ],
            ]);
    }

    public function create_mission_2x2(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/planetarium.png')
            ->with_media(MediaCollection::placeholder_complete, __DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/planetarium_completed.png')
            ->with_epilogue(
                text: '<p>This Sun is really shining, don’t get too close partner! Now…what else should we inspect… that treasure chest sure seems kind of odd. There’s an incomplete code written on it… 1 U 9 R … maybe we can use the code on the mirror to complete it.</p>',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Solve the planetarium',
                'description' => 'The children click on the small planetarium; the second test of the room is triggered (TEST 2). Once the second test is completed, the planetarium will appear with the celestial objects in different positions: the Sun globe will be positioned above all the other objects and start glowing. The heat from the sun causes the mirror located above it to start reacting to it. The sequence ” 7 X “ appears on the mirror as if it was burned on it.',
                'order' => 2,
                'pos_x' => 384,
                'pos_y' => 819,
                'configuration' => [
                    'hide_when_completed' => false,
                ],
            ]);
    }

    public function create_mission_2x3(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/mirror.png')
            ->with_media(MediaCollection::placeholder_complete, __DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/mirror_completed.png')
            ->create([
                'title' => 'Mirror placeholder',
                'description' => 'This mission is just a placeholder intended to update its image when the planetarium is solved.',
                'order' => 3,
                'pos_x' => 395,
                'pos_y' => 160,
                'configuration' => [
                    'complete_on_activation' => true,
                    'hide_when_completed' => false,
                    'hide_when_not_active' => false,
                ],
            ]);
    }

    public function create_mission_2x4(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/chest.png')
            ->with_media(MediaCollection::placeholder_complete, __DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/chest_completed.png')
            ->with_epilogue(
                text: '<p>Yes! The digits on the mirror were the missing digits to the secret code! Now what’s in the chest…a signal blocking device, a star shaped medallion and a what’s this? A Broken nametag? It says… TER. Might be useful in the following rooms. I think we can fit the star shaped medallion somewhere in the room…</p>',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Open the chest',
                'description' => 'The children select the treasure chest on the far-right corner of the room; the third test of the room is triggered (TEST 3). On completion of test 3, the treasure chest opens, and the children find the following items: A signal blocking device, A star shaped medallion, A piece of a broken nametag containing the letters “ TER “. The objects will be displayed in the UI of the game at the top left corner.',
                'order' => 4,
                'pos_x' => 1832,
                'pos_y' => 852,
                'configuration' => [
                    'hide_when_completed' => false,
                ],
            ]);
    }

    public function create_mission_2x5(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/medallion.png')
            ->with_epilogue(
                text: '<p>Great thinking partner! The medallion fits perfectly in the pedestal! The Keypad seems to be working now, let’s try interacting with it</p>',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Place the medallion',
                'description' => 'The children will have to click on the star shaped medallion, this triggers the fourth test of the room (TEST 4). Once the test is completed, the numerical keypad connected to the door that leads to the next room turns on.',
                'order' => 5,
                'pos_x' => 0,
                'pos_y' => 0,
                'configuration' => [
                    'hide_when_completed' => true,
                    'hide_when_not_active' => true,
                ],
            ]);
    }

    public function create_mission_2x6(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/column.png')
            ->with_media(MediaCollection::placeholder_complete, __DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/column_completed.png')
            ->create([
                'title' => 'Column placeholder',
                'description' => 'This mission is just a placeholder intended to update its image when the medallion is clicked.',
                'order' => 6,
                'pos_x' => 1210,
                'pos_y' => 814,
                'configuration' => [
                    'complete_on_activation' => true,
                    'hide_when_completed' => false,
                    'hide_when_not_active' => false,
                ],
            ]);
    }

    public function create_mission_2x7(Map $map): void
    {
        /** @var class-string<Mission> $mission_class */
        $mission_class = game_engine()->missionClass();

        $mission_class::factory()
            ->for($map)
            ->with_placeholder_image(__DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/keypad.png')
            ->with_media(MediaCollection::placeholder_active, __DIR__.'/../../resources/fixtures/test_story/maps/1_first_room/keypad_active.png')
            ->with_epilogue(
                text: '<p>Eureka! First room cleared my friend, now let’s move on to the second one! </p>',
                background_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/background.png',
                avatar_path: __DIR__.'/../../resources/fixtures/test_story/storytelling/avatar.png'
            )
            ->create([
                'title' => 'Click the keypad',
                'description' => 'The children click on the numerical keypad connected to the door, activating the fifth and final test of the room (TEST 5). Completion of the final test actives the self-evaluation sequence.',
                'order' => 7,
                'pos_x' => 1009,
                'pos_y' => 500,
                'configuration' => [
                    'hide_when_completed' => false,
                    'hide_when_not_active' => false,
                ],
            ]);
    }
}
