<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are prohibited without the prior written permission of DEF STUDIO S.R.L. This software is provided "as is" and any express or implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

/** @noinspection PhpDuplicateMatchArmBodyInspection */

namespace DefStudio\GameEngine\Models\Concerns;

use Spatie\Image\Manipulations;
use Spatie\MediaLibrary\InteractsWithMedia;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\Enums\MediaConversion;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

trait HasMedia
{
    use InteractsWithMedia {
        getMedia as getMediaFromSpatie;
        getFirstMedia as getFirstMediaFromSpatie;
        getFirstMediaUrl as getFirstMediaUrlFromSpatie;
    }

    private function register_media_collection(MediaCollection $collection): void
    {
        match ($collection) {
            MediaCollection::icon => $this
                ->addMediaCollection(MediaCollection::icon->name)
                ->singleFile()
                ->withResponsiveImages()
                ->registerMediaConversions(function(): void {
                    $this->register_standard_media_conversions();
                }),
            MediaCollection::background => $this
                ->addMediaCollection(MediaCollection::background->name)
                ->singleFile()
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::thumb->name)
                        ->nonQueued()
                        ->crop(Manipulations::CROP_CENTER, 64, 64)
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);

                    $this->addMediaConversion(MediaConversion::optimized->name)
                        ->nonQueued()
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
            MediaCollection::avatar => $this
                ->addMediaCollection(MediaCollection::avatar->name)
                ->singleFile()
                ->withResponsiveImages()
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::thumb->name)
                        ->nonQueued()
                        ->crop(Manipulations::CROP_CENTER, 64, 64)
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                    $this->addMediaConversion(MediaConversion::optimized->name)
                        ->nonQueued()
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
            MediaCollection::placeholder => $this
                ->addMediaCollection(MediaCollection::placeholder->name)
                ->singleFile()
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::optimized->name)
                        ->nonQueued()
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
            MediaCollection::placeholder_active => $this
                ->addMediaCollection(MediaCollection::placeholder_active->name)
                ->singleFile()
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::optimized->name)
                        ->nonQueued()
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
            MediaCollection::placeholder_completed => $this
                ->addMediaCollection(MediaCollection::placeholder_completed->name)
                ->singleFile()
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::optimized->name)
                        ->nonQueued()
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
            MediaCollection::cover => $this
                ->addMediaCollection(MediaCollection::cover->name)
                ->singleFile()
                ->withResponsiveImages()
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::thumb->name)
                        ->nonQueued()
                        ->crop(Manipulations::CROP_CENTER, 64, 64)
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                    $this->addMediaConversion(MediaConversion::square->name)
                        ->nonQueued()
                        ->crop(Manipulations::CROP_CENTER, 256, 256)
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
            MediaCollection::assets => $this
                ->addMediaCollection(MediaCollection::assets->name)
                ->withResponsiveImages()
                ->registerMediaConversions(function(): void {
                    $this->register_standard_media_conversions();
                }),
            MediaCollection::base_image => $this
                ->addMediaCollection(MediaCollection::base_image->name)
                ->singleFile()
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::thumb->name)
                        ->nonQueued()
                        ->crop(Manipulations::CROP_CENTER, 64, 64)
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
            MediaCollection::tile_placeholders => $this
                ->addMediaCollection(MediaCollection::tile_placeholders->name)
                ->registerMediaConversions(function(): void {
                    $this->addMediaConversion(MediaConversion::thumb->name)
                        ->nonQueued()
                        ->crop(Manipulations::CROP_CENTER, 64, 64)
                        ->quality(80)
                        ->format(Manipulations::FORMAT_WEBP);
                }),
        };
    }

    public function enable_media_collection(MediaCollection $collection): void
    {
        $this->media_collections = collect($this->media_collections)
            ->push($collection)
            ->unique()
            ->toArray();
    }

    public function registerMediaCollections(): void
    {
        collect($this->media_collections)
            ->map(fn(MediaCollection $collection) => $this->register_media_collection($collection));
    }

    public function getMedia(MediaCollection|string $collection = 'default', array|callable $filters = []): \Spatie\MediaLibrary\MediaCollections\Models\Collections\MediaCollection
    {
        $collection = $collection instanceof MediaCollection ? $collection->name : $collection;

        return $this->getMediaFromSpatie($collection, $filters);
    }

    public function getFirstMedia(MediaCollection|string $collection, $filters = []): ?Media
    {
        $collection = $collection instanceof MediaCollection ? $collection->name : $collection;

        return $this->getFirstMediaFromSpatie($collection, $filters);
    }

    public function getFirstMediaUrl(MediaCollection|string $collection, MediaConversion|string $conversion = null): string
    {
        $collection = $collection instanceof MediaCollection ? $collection->name : $collection;
        $conversion = $conversion instanceof MediaConversion ? $conversion->name : $conversion;

        return $this->getFirstMediaUrlFromSpatie($collection, $conversion ?? '');
    }

    private function register_standard_media_conversions(): void
    {
        $this->addMediaConversion(MediaConversion::thumb->name)
            ->nonQueued()
            ->crop(Manipulations::CROP_CENTER, 64, 64)
            ->quality(80)
            ->format(Manipulations::FORMAT_WEBP);
        $this->addMediaConversion(MediaConversion::small->name)
            ->crop(Manipulations::CROP_CENTER, 128, 128)
            ->quality(80)
            ->format(Manipulations::FORMAT_WEBP);
        $this->addMediaConversion(MediaConversion::medium->name)
            ->crop(Manipulations::CROP_CENTER, 256, 256)
            ->quality(80)
            ->format(Manipulations::FORMAT_WEBP);
        $this->addMediaConversion(MediaConversion::optimized->name)
            ->nonQueued()
            ->quality(80)
            ->format(Manipulations::FORMAT_WEBP);
    }
}
