<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

/** @noinspection PhpUnused */

namespace DefStudio\GameEngine\Decorators\Tasks;

use Illuminate\Support\Str;
use DefStudio\GameEngine\Http\Livewire\Tasks\Edit;
use DefStudio\GameEngine\Attributes\CallableFromLivewire;

class MultipleChoiceQuestion extends TaskDecorator
{
    public static function type_label(): string
    {
        return __('Multiple Choice Question');
    }

    public static function configuration_component(): string
    {
        return 'game-engine::tasks.multiple-choice-question.configuration';
    }

    public static function run_component(): string
    {
        return 'game-engine::tasks.multiple-choice-question.run';
    }

    public function question(): array
    {
        return $this->configuration['question'] ?? [];
    }

    public function answers(): array
    {
        return $this->configuration['answers'] ?? [];
    }

    public function should_randomize_answers_order(): bool
    {
        return $this->configuration['randomize_answers_order'] ?? false;
    }

    #[CallableFromLivewire]
    public function add_answer(): self
    {
        $this->configuration['answers'][] = [
            'id' => Str::uuid(),
            'answer' => [],
            'correct' => false,
        ];

        return $this;
    }

    /** @noinspection PhpUnusedParameterInspection */
    #[CallableFromLivewire]
    public function delete_answer(Edit $component, int $index): self
    {
        unset($this->configuration['answers'][$index]);

        return $this;
    }

    public function is_complete(array $state): bool
    {
        return !empty($state['answer']);
    }

    public function get_answer(string $id): array
    {
        $answer = collect($this->answers())->first(fn(array $answer) => $answer['id'] === $id);

        throw_if($answer === null, 'Invalid answer ID');

        return $answer;
    }

    public function get_answer_for_run(array $state): ?array
    {
        $answer_id = $state['result']['answer'] ?? null;

        if ($answer_id === null) {
            return null;
        }

        return $this->get_answer($answer_id);
    }

    public function process_completion(array $state): array
    {
        $answer = $this->get_answer_for_run($state);

        if ($answer === null) {
            return data_set($state, 'result.correct', false);
        }

        return data_set($state, 'result.correct', $answer['correct'] ?? false);
    }

    public function get_awards_for_run(array $state): array
    {
        $answer = $this->get_answer_for_run($state);

        return $answer['awards'] ?? [];
    }
}
