<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Actions\Missions\TaskExtractors;

use Illuminate\Support\Collection;
use DefStudio\GameEngine\Decorators\Tasks\TaskDecorator;
use DefStudio\GameEngine\Decorators\Runs\PlayableMission;
use DefStudio\GameEngine\Actions\Missions\Contracts\TaskExtractor;

abstract class TaskExtractorByLevel implements TaskExtractor
{
    private ?int $seed = null;

    abstract protected function current_level(PlayableMission $mission): int|string;

    public function extract(PlayableMission $mission): Collection
    {
        $tasks = $mission->tasks
            ->filter(fn(TaskDecorator $decorator) => $decorator->has_tag('level', $this->current_level($mission)));

        return $tasks
            ->when(
                $mission->should_randomize_tasks_order(),
                fn(Collection $tasks) => $tasks->shuffle($this->seed)
            )
            ->when(
                $mission->max_tasks_to_show(),
                fn(Collection $tasks, int $max) => $tasks->take($max)
            );
    }

    public function task_tags(): array
    {
        return [
            'level' => [
                'label' => __('Levels'),
                'tags' => collect(range(1, 3))
                    ->mapWithKeys(fn(int $level): array => [$level => __('Level :level', ['level' => $level])])
                    ->toArray(),
            ],
        ];
    }

    public function with_seed(?int $seed): static
    {
        $this->seed = $seed;

        return $this;
    }
}
