<?php

namespace DefStudio\GameEngine\Actions\Runs;

use Illuminate\Support\Collection;
use DefStudio\GameEngine\Decorators\Runs\PlayableMap;
use DefStudio\GameEngine\Decorators\Runs\PlayableTask;
use DefStudio\GameEngine\Decorators\Runs\PlayableStory;
use DefStudio\GameEngine\Decorators\Runs\PlayableMission;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;
use DefStudio\GameEngine\Actions\Runs\Contracts\AwardsEngine;

class DefaultAwardsEngine implements AwardsEngine
{
    public function compute_starting_awards(PlayableActivity $activity): void
    {
        // no starting awards by default
    }

    public function assign_awards(PlayableActivity $ended_activity): void
    {
        match ($ended_activity::class) {
            PlayableTask::class => $this->assign_task_awards($ended_activity),
            PlayableMission::class => $this->assign_mission_awards($ended_activity),
            PlayableMap::class => $this->assign_map_awards($ended_activity),
            PlayableStory::class => $this->assign_story_awards($ended_activity),
            default => null,
        };
    }

    protected function assign_task_awards(PlayableTask $task): void
    {
        $task->set_state('result.awards', $this->merge_awards(
            $task->get_state('result.base_awards', []),
            $task->get_awards_for_run($task->get_state()),
        ));
        $this->assign_mission_awards($task->playable_mission());
    }

    protected function assign_mission_awards(PlayableMission $mission): void
    {
        $mission->set_state('result.awards', $this->merge_awards(
            $mission->get_state('result.base_awards', []),
            $this->extract_children_awards($mission->playable_tasks()),
        ));
        $this->assign_map_awards($mission->playable_map());
    }

    public function set_base_awards(PlayableActivity $activity, array $awards): void
    {
        $activity->set_state('result.base_awards', $awards);
    }

    protected function assign_map_awards(PlayableMap $map): void
    {
        $map->set_state('result.awards', $this->merge_awards(
            $map->get_state('result.base_awards', []),
            $this->extract_children_awards($map->playable_missions()),
        ));

        $this->assign_story_awards($map->playable_story());
    }

    protected function assign_story_awards(PlayableStory $story): void
    {
        $awards = $this->merge_awards(
            $story->get_state('result.base_awards', []),
            $this->extract_children_awards($story->playable_maps()),
        );

        $story->set_state('result.awards', $awards);

        $story->run()->awards = $awards;
    }

    /**
     * @param  Collection<int, PlayableActivity>  $children
     * @return array<int, float>
     */
    protected function extract_children_awards(Collection $children): array
    {
        return $children->reduce(function(array $awards, PlayableActivity $activity): array {
            foreach ($activity->get_state('result.awards', []) as $award_id => $awarded_quantity) {
                if (!isset($awards[$award_id])) {
                    $awards[$award_id] = 0;
                }

                $awards[$award_id] += (float) $awarded_quantity;
            }

            return $awards;
        }, []);
    }

    protected function merge_awards(array ...$award_sets): array
    {
        $awards = [];

        foreach ($award_sets as $award_set) {
            foreach ($award_set as $award_id => $award_value) {
                if (!isset($awards[$award_id])) {
                    $awards[$award_id] = 0;
                }

                $awards[$award_id] += $award_value;
            }
        }

        return $awards;
    }
}
