<?php

/** @noinspection PhpUndefinedMethodInspection */

namespace DefStudio\GameEngine\Http\Livewire\Organizations;

use Livewire\Component;
use Illuminate\Contracts\View\View;
use DefStudio\GameEngine\Enums\Role;
use Illuminate\Support\Facades\Auth;
use DefStudio\GameEngine\Models\User;
use Illuminate\Support\Facades\Route;
use Illuminate\Validation\Rules\Password;
use DefStudio\GameEngine\Models\Organization;
use DefStudio\GameEngine\Actions\Users\SaveUser;
use DefStudio\GameEngine\Actions\Organizations\SaveOrganization;

class Register extends Component
{
    public string $admin_name = '';

    public string $admin_email = '';

    public string $admin_password = '';

    public bool $terms = false;

    public string $organization_name = '';

    public function rules(): array
    {
        $rules = [
            'admin_name' => 'required',
            'admin_email' => 'required|email|unique:users,email',
            'admin_password' => ['required', $this->password_rules()],
            'organization_name' => 'required',
        ];

        if (Route::has('policy.show')) {
            $rules['terms'] = 'accepted';
        }

        return $rules;
    }

    protected function password_rules(): Password
    {
        return Password::min(8)->numbers()->letters()->symbols()->uncompromised()->mixedCase();
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.organizations.register');
    }

    public function register(): void
    {
        $this->validate();

        $organization = $this->make_organization();
        SaveOrganization::run($organization, authorize: false);

        $user = $this->make_user();
        SaveUser::run($user, password: $this->admin_password, authorize: false);

        Auth::logout();
        Auth::login($user);

        $user->subscribe_to_organization($organization, Role::admin, true);

        $this->redirect(\route(config('game-engine.organizations.register_organization.redirect_to_route')));
    }

    protected function make_user(): User
    {
        return game_engine()->userQuery()->make([
            'name' => $this->admin_name,
            'email' => $this->admin_email,
        ]);
    }

    public function make_organization(): Organization
    {
        return game_engine()->organizationQuery()->make([
            'name' => $this->organization_name,
        ]);
    }
}
