<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Http\Livewire\Runs;

use Livewire\Component;
use Illuminate\Contracts\View\View;
use DefStudio\GameEngine\Models\Map;
use Illuminate\Support\Facades\Gate;
use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Models\Mission;
use DefStudio\GameEngine\Enums\Permission;
use DefStudio\GameEngine\Models\Storytelling;
use DefStudio\GameEngine\Decorators\Tasks\TaskDecorator;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\RunsMaps;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\RunsStory;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\OpensModals;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\RunsMissions;
use DefStudio\GameEngine\Http\Livewire\Concerns\CallsTraitsMethods;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\RunsStorytellings;
use DefStudio\GameEngine\Http\Livewire\Runs\Concerns\RetrievesActivities;

class Runner extends Component
{
    use AuthorizesRequests;
    use CallsTraitsMethods;
    use OpensModals;
    use RetrievesActivities;
    use RunsMaps;
    use RunsMissions;
    use RunsStory;
    use RunsStorytellings;

    protected $listeners = [
        'next_mission',
        'complete_mission',
        'complete_storytelling',
    ];

    public int $run_id;

    public array $images_to_preload = [];

    public function mount(): void
    {
        $this->authorize('play', $this->run);

        if (Feature::preload_images->enabled()) {
            $this->images_to_preload = $this->story->loadMissing([
                'maps.media',
                'maps.storytellings.media',
                'maps.missions.media',
                'maps.missions.storytellings.media',
                'maps.missions.tasks.media',
                'maps.missions.tasks.storytellings.media',
            ])
                ->maps->map(fn(Map $map) => [
                    $map->all_media_urls(),
                    $map->storytellings->map(fn(Storytelling $storytelling) => $storytelling->all_media_urls()),
                    $map->missions->map(fn(Mission $mission) => [
                        $mission->all_media_urls(),
                        $mission->storytellings->map(fn(Storytelling $storytelling) => $storytelling->all_media_urls()),
                        $mission->tasks->map(fn(TaskDecorator $task) => [
                            $task->all_media_urls(),
                            $task->storytellings->map(fn(Storytelling $storytelling) => $storytelling->all_media_urls()),
                        ]),
                    ]),
                ])->flatten()->filter()->unique()->toArray();
        }
    }

    public function booted(): void
    {
        $this->story->tick_timer();

        if ($this->is_map_active()) {
            $this->map->tick_timer();
        }

        if ($this->is_mission_active()) {
            $this->mission->tick_timer();
        }

        $this->save();
    }

    public function render(): View
    {
        return view('game-engine::livewire.runs.runner');
    }

    public function reset_run(): void
    {
        Gate::authorize(Permission::debug_runs->value);

        $this->run->reset_state();
        $this->run->completed_at = null;
        $this->run->started_at = null;
        $this->save();

        $this->reload_page();
    }

    protected function reload_page(): void
    {
        $this->redirect(route('game-engine.run', $this->run));
    }

    protected function save(): void
    {
        $this->run->save();
    }
}
