<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Policies;

use DefStudio\GameEngine\Enums\Role;
use DefStudio\GameEngine\Models\User;
use DefStudio\GameEngine\Models\Story;
use DefStudio\GameEngine\Enums\Permission;
use Illuminate\Auth\Access\HandlesAuthorization;

class UserPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user): bool
    {
        if ($user->can(Permission::view_users->value)) {
            return true;
        }

        foreach (Role::cases() as $role) {
            if ($user->can(Permission::view_users->for_role($role))) {
                return true;
            }
        }

        return false;
    }

    public function view(User $user, User $target): bool
    {
        return $user->can(Permission::view_users->for_role(
            $target->get_role_in_organization($user->current_organization)
        ));
    }

    public function update(User $user, User $target): bool
    {
        return $user->can(Permission::edit_users->for_role(
            $target->get_role_in_organization($user->current_organization)
        ));
    }

    public function delete(User $user, User $target): bool
    {
        return $user->can(Permission::delete_users->for_role(
            $target->get_role_in_organization($user->current_organization)
        ));
    }

    public function impersonate(User $user, User $target): bool
    {
        if (!$user->can_impersonate()) {
            return false;
        }

        return $target->can_be_impersonated();
    }

    public function create_run(User $user, User $target_user, ?Story $story = null): bool
    {
        if ($story !== null && !$target_user->can('play', $story)) {
            return false;
        }

        if (!$user->can(Permission::create_runs->value)) {
            return false;
        }

        return $user->can('view', $target_user);
    }
}
