<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection LaravelUnknownEloquentFactoryInspection */

namespace DefStudio\GameEngine\Models;

use Carbon\Carbon;
use Spatie\MediaLibrary\HasMedia;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\Attributes\ParentScope;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use DefStudio\GameEngine\Models\Contracts\FitsInScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use DefStudio\GameEngine\Models\Concerns\DefinesAwards;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use DefStudio\GameEngine\Models\Concerns\BelongsToScope;
use DefStudio\GameEngine\Models\Contracts\AwardsDefiner;
use DefStudio\GameEngine\Database\Factories\StoryFactory;
use DefStudio\GameEngine\Models\Concerns\HasStorytellings;
use DefStudio\GameEngine\Models\Contracts\WithStorytelling;
use DefStudio\GameEngine\Models\Concerns\CanCheckRelationships;

/**
 * @property int $id
 * @property string $title
 * @property string $description
 * @property Carbon|null $published_at
 * @property int $organization_id
 * @property-read Organization $organization
 * @property-read Collection<int, Team> $teams
 * @property-read Collection<int, Map> $maps
 *
 * @method static StoryFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
class Story extends Model implements AwardsDefiner, FitsInScope, HasMedia, WithStorytelling
{
    use BelongsToScope;
    use CanCheckRelationships;
    use Concerns\HasMedia;
    use DefinesAwards;
    use HasFactory;
    use HasStorytellings;

    protected $fillable = [
        'title',
        'description',
        'published_at',
    ];

    protected $casts = [
        'published_at' => 'datetime',
    ];

    protected array $media_collections = [
        MediaCollection::cover,
    ];

    public function getTable()
    {
        return config('game-engine.stories.table', parent::getTable());
    }

    protected static function newFactory(): StoryFactory
    {
        return StoryFactory::new();
    }

    protected static function booted()
    {
        self::deleting(function(Story $model): void {
            $model->teams()->detach();
            $model->users()->detach();
        });
    }

    /**
     * @return BelongsTo<Organization, Story>
     */
    #[ParentScope(Organization::class)]
    public function organization(): BelongsTo
    {
        return $this->belongsTo(\game_engine()->organizationClass());
    }

    public function maps(): HasMany
    {
        return $this->hasMany(Map::class);
    }

    public function award_scope_label(): string
    {
        return __('Story: :story_title', ['story_title' => $this->title]);
    }

    /**
     * @return MorphToMany<Team>
     */
    public function teams(): MorphToMany
    {
        return $this->morphedByMany(game_engine()->teamClass(), 'subscriber', config('game-engine.stories.subscriptions_table'));
    }

    /**
     * @return MorphToMany<User>
     */
    public function users(): MorphToMany
    {
        return $this->morphedByMany(game_engine()->userClass(), 'subscriber', config('game-engine.stories.subscriptions_table'));
    }

    public function scopePublished(Builder $query): void
    {
        $query->whereNotNull('published_at');
    }

    public function scopeUnpublished(Builder $query): void
    {
        $query->whereNull('published_at');
    }

    public function snapshot(): array
    {
        return [
            'id' => $this->id,
            'title' => $this->title,
            'maps' => $this->maps
                ->sortBy('order')
                ->map(fn(Map $map) => $map->snapshot()),
        ];
    }
}
