<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Decorators\Runs;

use DefStudio\GameEngine\Models\Run;
use Illuminate\Database\Eloquent\Model;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Decorators\ModelDecorator;
use DefStudio\GameEngine\Exceptions\RunnerException;
use DefStudio\GameEngine\Exceptions\StorytellingException;

/**
 * @template TModel of Model|ModelDecorator
 *
 * @property-read int $id
 *
 * @extends ModelDecorator<TModel>
 *
 * @mixin Model
 */
abstract class PlayableActivity extends ModelDecorator
{
    protected Run $run;

    public static string $state_key;

    public function run(): Run
    {
        return $this->run;
    }

    public function setup(Run $run): static
    {
        $this->run = $run;

        return $this;
    }

    public function state_key(): string
    {
        return static::$state_key;
    }

    /**
     * @return ($key is null ? array<string, mixed> : mixed)
     *
     * @noinspection PhpMissingReturnTypeInspection
     */
    public function get_state(string $key = null, mixed $default = null)
    {
        if ($key === null) {
            return $this->run->get_state("{$this->state_key()}.$this->id", $default ?? []);
        }

        return $this->run->get_state("{$this->state_key()}.$this->id.$key", $default);
    }

    public function replace_state(array $new_state): void
    {
        $this->run->set_state("{$this->state_key()}.$this->id", $new_state);
    }

    public function set_state(string $key, mixed $value): void
    {
        $this->run->set_state("{$this->state_key()}.$this->id.$key", $value);
    }

    public function forget_state(string $key): void
    {
        $this->run->forget_state("{$this->state_key()}.$this->id.$key");
    }

    public function append_state(string $key, mixed $value): void
    {
        $state = $this->get_state($key, []);

        if (!is_array($state)) {
            throw RunnerException::cannot_append_state($key);
        }

        $state[] = $value;

        $this->set_state($key, $state);
    }

    public function start_timer(): void
    {
        if (!empty($current_timers = $this->get_state('timers.current', []))) {
            $this->append_state('timers.backups', $current_timers);
            $this->set_state('timers.current', []);
        }

        $this->set_state('timers.current.start', (string) now());
    }

    public function tick_timer(): void
    {
        if ($this->get_state('timers.current.start') === null) {
            return;
        }

        $this->append_state('timers.current.ticks', (string) now());
    }

    public function stop_timer(): void
    {
        if ($this->get_state('timers.current.start') === null) {
            return;
        }

        if ($this->get_state('timers.current.end') !== null) {
            return;
        }

        $this->set_state('timers.current.end', (string) now());
    }

    public function setup_storytelling(): void
    {
        if ($this->get_state('storytelling') !== null) {
            return;
        }

        foreach (StorytellingType::cases() as $storytelling_type) {
            /** @noinspection PhpUndefinedFieldInspection */
            /** @phpstan-ignore-next-line  */
            $this->set_state("storytelling.$storytelling_type->value", $this->storytellings->where('type', $storytelling_type)->isEmpty());
        }

        $this->run->save();
    }

    public function complete_storytelling(StorytellingType $type): void
    {
        $this->set_state("storytelling.$type->value", true);
        $this->run->save();
    }

    public function has_pending_storytellings(StorytellingType $type = null): bool
    {
        if ($type === null) {
            return collect($this->get_state('storytelling', []))
                ->reject(fn(bool $complete) => $complete)->isNotEmpty();
        }

        $done = $this->get_state("storytelling.$type->value");

        throw_if($done === null, StorytellingException::not_set_up($this));

        return !$done;
    }
}
