<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Missions;

use DefStudio\GameEngine\Enums\TimeUnit;
use DefStudio\GameEngine\Http\Livewire\Concerns\ManagesTimeLimit;
use Livewire\WithFileUploads;
use Illuminate\Contracts\View\View;
use Livewire\TemporaryUploadedFile;
use Illuminate\Support\Facades\Gate;
use LivewireUI\Modal\ModalComponent;
use Illuminate\Validation\Rules\File;
use DefStudio\GameEngine\Models\Mission;
use Illuminate\Validation\ValidationException;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\Actions\Tasks\DeleteTask;
use DefStudio\GameEngine\Actions\Missions\UpdateMission;
use DefStudio\GameEngine\Actions\Missions\StoreCoverImage;
use DefStudio\GameEngine\Actions\Missions\RemoveCoverImage;
use DefStudio\GameEngine\Http\Livewire\Concerns\ImprovedModal;
use DefStudio\GameEngine\Http\Livewire\Concerns\RedirectsBack;
use DefStudio\GameEngine\Actions\Livewire\Sortable\UpdateOrder;
use DefStudio\GameEngine\Actions\Missions\StoreBackgroundImage;
use DefStudio\GameEngine\Actions\Missions\RemoveBackgroundImage;
use DefStudio\GameEngine\Actions\Missions\StorePlaceholderImage;
use DefStudio\GameEngine\Actions\Missions\RemovePlaceholderImage;
use DefStudio\GameEngine\Actions\Missions\StorePlaceholderActiveImage;
use DefStudio\GameEngine\Actions\Missions\RemovePlaceholderActiveImage;
use DefStudio\GameEngine\Actions\Missions\StorePlaceholderCompletedImage;
use DefStudio\GameEngine\Actions\Missions\RemovePlaceholderCompletedImage;

class Edit extends ModalComponent
{
    use ImprovedModal;
    use RedirectsBack;
    use WithFileUploads;
    use ManagesTimeLimit;

    /** @var Mission */
    public $mission;

    /** @var TemporaryUploadedFile|null */
    public $cover_image = null;

    /** @var TemporaryUploadedFile|null */
    public $background_image = null;

    /** @var TemporaryUploadedFile|null */
    public $placeholder_image = null;

    /** @var TemporaryUploadedFile|null */
    public $placeholder_completed_image = null;

    /** @var TemporaryUploadedFile|null */
    public $placeholder_active_image = null;

    public array $rules = [
        'mission.title' => 'required|string|max:255',
        'mission.description' => 'nullable|string',
        'mission.configuration.randomize_tasks_order' => 'boolean|nullable',
        'mission.configuration.max_tasks_to_show' => 'integer|nullable',
        'mission.configuration.hide_when_not_active' => 'boolean|nullable',
        'mission.configuration.hide_when_completed' => 'boolean|nullable',
        'mission.configuration.autostart_on_activation' => 'boolean|nullable',
        'mission.configuration.complete_on_activation' => 'boolean|nullable',
        'mission.configuration.always_active' => 'boolean|nullable',
    ];

    protected $listeners = [
        'task.saved' => 'refresh_tasks',
    ];

    public function time_limit_field(): string
    {
        return 'mission.configuration.time_limit';
    }

    public function mount(int $mission_id): void
    {
        $this->mission = game_engine()->missionQuery()->find($mission_id);

        Gate::authorize('update', $this->mission);
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.missions.edit');
    }

    public function updatedCoverImage(): void
    {
        if ($this->cover_image === null) {
            return;
        }

        try {
            $this->validate(['cover_image' => File::image()]);
        } catch (ValidationException $validationException) {
            $this->cover_image = null;
            throw $validationException;
        }
    }

    public function remove_cover_image(): void
    {
        $this->cover_image = null;
        RemoveCoverImage::run($this->mission);
    }

    public function updatedBackgroundImage(): void
    {
        if ($this->background_image === null) {
            return;
        }

        try {
            $this->validate(['background_image' => File::image()]);
        } catch (ValidationException $validationException) {
            $this->background_image = null;
            throw $validationException;
        }
    }

    public function updatedPlaceholderImage(): void
    {
        if ($this->placeholder_image === null) {
            return;
        }

        try {
            $this->validate(['placeholder_image' => File::image()]);
        } catch (ValidationException $validationException) {
            $this->placeholder_image = null;
            throw $validationException;
        }
    }

    public function updatedPlaceholderActiveImage(): void
    {
        if ($this->placeholder_active_image === null) {
            return;
        }

        try {
            $this->validate(['placeholder_active_image' => File::image()]);
        } catch (ValidationException $validationException) {
            $this->placeholder_active_image = null;
            throw $validationException;
        }
    }

    public function updatedPlaceholderCompletedImage(): void
    {
        if ($this->placeholder_completed_image === null) {
            return;
        }

        try {
            $this->validate(['placeholder_completed_image' => File::image()]);
        } catch (ValidationException $validationException) {
            $this->placeholder_completed_image = null;
            throw $validationException;
        }
    }

    public function remove_background_image(): void
    {
        $this->background_image = null;
        RemoveBackgroundImage::run($this->mission);
    }

    public function remove_placeholder_image(): void
    {
        $this->placeholder_image = null;
        RemovePlaceholderImage::run($this->mission);
    }

    public function remove_placeholder_active_image(): void
    {
        $this->placeholder_active_image = null;
        RemovePlaceholderActiveImage::run($this->mission);
    }

    public function remove_placeholder_completed_image(): void
    {
        $this->placeholder_completed_image = null;
        RemovePlaceholderCompletedImage::run($this->mission);
    }

    public function save(): void
    {
        $this->validate();

        if (!$this->mission->exists || $this->mission->getFirstMedia(MediaCollection::placeholder) === null) {
            $this->validate(['placeholder_image' => 'required']);
        }

        $this->store_time_limit();

        UpdateMission::run($this->mission);

        if ($this->cover_image !== null) {

            StoreCoverImage::run($this->mission, $this->cover_image);
        }

        if ($this->background_image !== null) {
            StoreBackgroundImage::run($this->mission, $this->background_image);
        }

        if ($this->placeholder_image !== null) {
            StorePlaceholderImage::run($this->mission, $this->placeholder_image);
        }

        if ($this->placeholder_active_image !== null) {
            StorePlaceholderActiveImage::run($this->mission, $this->placeholder_active_image);
        }

        if ($this->placeholder_completed_image !== null) {
            StorePlaceholderCompletedImage::run($this->mission, $this->placeholder_completed_image);
        }

        $this->emit('mission.updated', $this->mission->id);

        if ($this->isModal) {
            $this->closeModal();

            return;
        }

        $this->redirect_back();
    }

    public function refresh_tasks(): void
    {
        $this->mission->unsetRelation('tasks');
    }

    public function delete_task(int $task_id): void
    {
        $task = $this->mission->tasks()->findOrFail($task_id);

        DeleteTask::run($task);

        $this->mission->unsetRelation('tasks');
    }

    public function update_tasks_order(array $new_order): void
    {
        UpdateOrder::run($this->mission->tasks, $new_order);
    }
}
