<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection LaravelUnknownEloquentFactoryInspection */
/** @noinspection PhpPossiblePolymorphicInvocationInspection */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Models;

use BaconQrCode\Writer;
use Illuminate\Support\HtmlString;
use BaconQrCode\Renderer\Color\Rgb;
use Illuminate\Support\Facades\URL;
use DefStudio\GameEngine\Enums\Role;
use Illuminate\Validation\Rules\Password;
use Spatie\Permission\Traits\HasRoles;
use BaconQrCode\Renderer\ImageRenderer;
use DefStudio\GameEngine\Enums\Feature;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Builder;
use BaconQrCode\Renderer\RendererStyle\Fill;
use Illuminate\Database\Eloquent\Collection;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use DefStudio\Impersonator\Concerns\Impersonate;
use DefStudio\GameEngine\Models\Concerns\HasTeams;
use DefStudio\GameEngine\Exceptions\StoryException;
use Illuminate\Database\Eloquent\Relations\HasMany;
use DefStudio\GameEngine\Models\Concerns\HasStories;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use DefStudio\GameEngine\Database\Factories\UserFactory;
use Illuminate\Contracts\Translation\HasLocalePreference;
use DefStudio\GameEngine\Models\Concerns\HasOrganizations;
use DefStudio\GameEngine\Models\Contracts\SubscribesToStories;
use DefStudio\GameEngine\Models\Concerns\CanCheckRelationships;

/**
 * @property int $id
 * @property string $name
 * @property string $email
 * @property string $password
 * @property string $locale
 * @property int|null $current_organization_id
 * @property-read Collection<int, Team> $teams
 * @property-read Collection<int, Team> $current_teams
 * @property-read Collection<int, Organization> $organizations
 * @property-read Collection<int, Run> $runs
 * @property-read Collection<int, Run> $available_runs
 * @property-read Organization|null $current_organization
 *
 * @method static UserFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
abstract class User extends Authenticatable implements HasLocalePreference, SubscribesToStories
{
    use Notifiable;
    use HasFactory;
    use CanCheckRelationships;
    use HasOrganizations;
    use HasTeams;
    use HasStories;
    use HasRoles {
        HasRoles::hasRole as _hasRole;
        HasRoles::hasPermissionTo as _hasPermissionTo;
    }
    use Impersonate;

    protected $fillable = [
        'name',
        'email',
        'password',
        'locale',
    ];

    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    protected $attributes = [
        'locale' => 'en',
    ];

    protected $with = ['organizations'];

    /**
     * @noinspection PhpMissingReturnTypeInspection
     */
    protected static function newFactory()
    {
        return UserFactory::new();
    }

    public function preferredLocale(): string
    {
        return $this->locale;
    }

    public function can_be_impersonated(): bool
    {
        if (!Feature::impersonate->enabled()) {
            return false;
        }

        return !$this->hasRole(Role::super_admin->value);
    }

    public function can_impersonate(): bool
    {
        if (!Feature::impersonate->enabled()) {
            return false;
        }

        return $this->hasRole(Role::super_admin);
    }

    public function hasRole($roles, string $guard = null): bool
    {
        if ($roles instanceof Role) {
            $roles = $roles->value;
        }

        return $this->_hasRole($roles, $guard);
    }

    public function hasPermissionTo($permission, $guardName = null): bool
    {
        if ($permission instanceof Role) {
            $permission = $permission->value;
        }

        return $this->_hasPermissionTo($permission, $guardName);
    }

    /**
     * @return Collection<int, Story>
     */
    public function available_stories(): Collection
    {
        if ($this->current_organization_id === null) {
            return Collection::empty();
        }

        if ($this->hasRole(Role::super_admin)) {
            return $this->current_organization
                ->stories()
                ->get();
        }

        return $this->current_organization
            ->stories()
            ->published()
            ->where(fn(Builder $query) => $query
                ->whereRaw('1 = 0')
                ->when(
                    Feature::user_stories->enabled(),
                    fn(Builder $subquery) => $subquery->orWhereRelation('users', 'subscriber_id', $this->id))
                ->when(
                    Feature::team_stories->enabled(),
                    fn(Builder $subquery) => $subquery->orWhereRelation(
                        'teams',
                        fn(Builder $teams_query) => $teams_query->whereIn(
                            'subscriber_id',
                            $this->current_teams()->pluck('id')
                        )
                    )
                ))->get();
    }

    /**
     * @return HasMany<Run>
     */
    public function runs(): HasMany
    {
        return $this->hasMany(game_engine()->runClass());
    }

    /**
     * @return HasMany<Run>
     */
    public function available_runs(): HasMany
    {
        if ($this->current_organization_id === null) {
            return $this->runs()->whereRaw('1 = 0');
        }

        return $this->runs()
            ->where(function(Builder $query) {
                $query->whereNull('available_from_date')->orWhere('available_from_date', '<=', now());
            })
            ->where(function(Builder $query) {
                $query->whereNull('available_until_date')->orWhere('available_until_date', '>=', now());
            })->whereNull('completed_at');
    }

    public function stories_feature(): Feature
    {
        return Feature::user_stories;
    }

    public function ensure_can_subscribe_to_story(Story $story): void
    {
        throw_unless($this->belongs_to_organization($story->organization), StoryException::organization_membership_failure());
    }

    public function completed_story(Story $story): bool
    {
        return game_engine()->runQuery()->forUser($this)->ofStory($story)->completed()->count() > 0;
    }

    public function password_reset_qr_code(int $size = 192): HtmlString
    {
        $svg = (new Writer(
            new ImageRenderer(
                new RendererStyle($size, 0, null, null, Fill::uniformColor(new Rgb(255, 255, 255), new Rgb(45, 55, 72))),
                new SvgImageBackEnd
            )
        ))->writeString($this->password_reset_url());

        return str($svg)->after('\n')->trim()->toHtmlString();
    }

    public function password_reset_url(): string
    {
        throw_unless(config('game-engine.users.reset_password.enabled', false), 'Reset password is disabled');

        $validity_days = config('game-engine.users.reset_password.url_validity_days');

        if ($validity_days) {
            return URL::temporarySignedRoute('game-engine.reset-password', today()->endOfDay()->addDays(config('game-engine.users.reset_password.url_validity_days')), ['user' => $this]);
        }

        return URL::signedRoute('game-engine.reset-password', ['user' => $this]);
    }

    public static function password_rules(): Password
    {
        return  Password::default();
    }
}
