<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpPossiblePolymorphicInvocationInspection */

namespace DefStudio\GameEngine\Decorators\Runs;

use Illuminate\Support\Collection;
use DefStudio\GameEngine\Models\Mission;
use DefStudio\GameEngine\Enums\MediaCollection;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use DefStudio\GameEngine\Decorators\Tasks\TaskDecorator;

/**
 * @extends PlayableActivity<Mission>
 *
 * @mixin Mission
 */
class PlayableMission extends PlayableActivity
{
    public static string $state_key = 'missions';

    public function playable_map(): PlayableMap
    {
        $this->loadMissing('map');

        return PlayableMap::build($this->map)->setup($this->run);
    }

    /**
     * @return Collection<int, PlayableTask>
     */
    public function playable_tasks(): Collection
    {
        return $this->tasks
            ->filter(fn(TaskDecorator|PlayableTask $task) => in_array($task->id, $this->get_state('extracted_tasks')))
            ->map(fn(TaskDecorator|PlayableTask $task) => PlayableTask::build($task)->setup($this->run));
    }

    protected function setup_decorator(): void
    {
        $this->model->loadMissing(['media', 'storytellings', 'tasks']);
    }

    public function active(): bool
    {
        return $this->get_state('extracted_tasks') !== null;
    }

    public function completed(): bool
    {
        return $this->run->get_state("missions.$this->id.completed", false);
    }

    public function all_tasks_completed(): bool
    {
        if (!$this->active()) {
            return false;
        }

        return $this->playable_tasks()->every(fn(PlayableTask $task) => $task->completed());
    }

    public function placeholder(): Media
    {
        return $this->getFirstMedia(MediaCollection::placeholder);
    }

    public function placeholder_complete(): ?Media
    {
        return $this->getFirstMedia(MediaCollection::placeholder_completed) ?? $this->getFirstMedia(MediaCollection::placeholder);
    }

    public function placeholder_active(): ?Media
    {
        return $this->getFirstMedia(MediaCollection::placeholder_active) ?? $this->getFirstMedia(MediaCollection::placeholder);
    }

    public function current_placeholder(): Media
    {
        if ($this->completed()) {
            return $this->placeholder_complete();
        }

        if ($this->active()) {
            return $this->placeholder_active();
        }

        return $this->placeholder();
    }

    public function visible(): bool
    {
        if ($this->should_hide_when_not_active() && !$this->active()) {
            return false;
        }

        if (!$this->should_hide_when_completed()) {
            return true;
        }

        return !$this->completed();
    }
}
