<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection LaravelUnknownEloquentFactoryInspection */

namespace DefStudio\GameEngine\Models;

use Spatie\MediaLibrary\HasMedia;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use DefStudio\GameEngine\Enums\MediaCollection;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Casts\AsArrayObject;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use DefStudio\GameEngine\Models\Contracts\AwardsDefiner;
use DefStudio\GameEngine\Database\Factories\AwardFactory;

/**
 * @property int $id
 * @property string $name
 * @property string|null $description
 * @property array $configuration
 * @property string|null $scope_type
 * @property string|null $scope_id
 * @property int|null $parent_id
 * @property-read self|null $parent
 * @property-read Collection<int, self> $children
 * @property-read (AwardsDefiner&Model)|null $scope
 */
class Award extends Model implements HasMedia
{
    use HasFactory;
    use Concerns\HasMedia;

    protected $fillable = [
        'name',
        'description',
    ];

    protected $casts = [
        'configuration' => AsArrayObject::class,
        'order' => 'integer',
    ];

    protected $attributes = [
        'configuration' => '{}',
    ];

    protected array $media_collections = [
        MediaCollection::icon,
    ];

    public function getTable()
    {
        return config('game-engine.awards.table', parent::getTable());
    }

    protected static function newFactory(): AwardFactory
    {
        return AwardFactory::new();
    }

    /**
     * @return BelongsTo<self, self>
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(game_engine()->awardClass(), 'parent_id');
    }

    /**
     * @return HasMany<self>
     */
    public function children(): HasMany
    {
        return $this->hasMany(game_engine()->awardClass(), 'parent_id');
    }

    /**
     * @return MorphTo<Model, Award>
     */
    public function scope(): MorphTo
    {
        return $this->morphTo();
    }

    public function scopeGlobal(Builder $query): void
    {
        $query->whereNull('scope_id')
            ->whereNull('parent_id');
    }

    public function snapshot(): array
    {
        $this->loadMissing('children');

        return [
            'id' => $this->id,
            'name' => $this->name,
            'children' => $this->children->map(fn(Award $award) => $award->snapshot()),
        ];
    }
}
