<?php

namespace DefStudio\GameEngine\Actions\Runs;

use Illuminate\Support\Collection;
use DefStudio\GameEngine\Decorators\Runs\PlayableMap;
use DefStudio\GameEngine\Decorators\Runs\PlayableTask;
use DefStudio\GameEngine\Decorators\Runs\PlayableStory;
use DefStudio\GameEngine\Decorators\Runs\PlayableMission;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

class DefaultAwardsEngine implements \DefStudio\GameEngine\Actions\Runs\Contracts\AwardsEngine
{
    public function compute_starting_awards(PlayableStory $story): void
    {
        // no starting awards by default
    }

    public function assign_awards(PlayableActivity $ended_activity): void
    {
        if ($ended_activity instanceof PlayableTask) {
            $this->assign_task_awards($ended_activity);
            $this->assign_mission_awards($mission = $ended_activity->playable_mission());
            $this->assign_map_awards($map = $mission->playable_map());
            $this->assign_story_awards($map->playable_story());
        }

        if ($ended_activity instanceof PlayableMission) {
            $this->assign_mission_awards($ended_activity);
            $this->assign_map_awards($map = $ended_activity->playable_map());
            $this->assign_story_awards($map->playable_story());
        }

        if ($ended_activity instanceof PlayableMap) {
            $this->assign_map_awards($ended_activity);
            $this->assign_story_awards($ended_activity->playable_story());
        }

        if ($ended_activity instanceof PlayableStory) {
            $this->assign_story_awards($ended_activity);
        }
    }

    private function assign_task_awards(PlayableTask $task): void
    {
        $task->set_state('result.awards', $task->get_awards_for_run($task->get_state()));
    }

    private function assign_mission_awards(PlayableMission $mission): void
    {
        $mission->set_state('result.awards', $this->extract_children_awards($mission->playable_tasks()));
    }

    private function assign_map_awards(PlayableMap $map): void
    {
        $map->set_state('result.awards', $this->extract_children_awards($map->playable_missions()));
    }

    private function assign_story_awards(PlayableStory $story): void
    {
        $story->set_state('result.awards', $this->extract_children_awards($story->playable_maps()));
    }

    /**
     * @param  Collection<int, PlayableActivity>  $children
     * @return array<int, float>
     */
    private function extract_children_awards(Collection $children): array
    {
        return $children->reduce(function(array $awards, PlayableActivity $activity): array {
            foreach ($activity->get_state('result.awards', []) as $award_id => $awarded_quantity) {
                if (!isset($awards[$award_id])) {
                    $awards[$award_id] = 0;
                }

                $awards[$award_id] += (float) $awarded_quantity;
            }

            return $awards;
        }, []);
    }
}
