<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnused */

/** @noinspection LaravelUnknownEloquentFactoryInspection */

namespace DefStudio\GameEngine\Models;

use ArrayObject;
use Spatie\MediaLibrary\HasMedia;
use Illuminate\Database\Eloquent\Model;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\Attributes\ParentScope;
use Illuminate\Database\Eloquent\Casts\AsArrayObject;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use DefStudio\GameEngine\Models\Contracts\FitsInScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use DefStudio\GameEngine\Database\Factories\TaskFactory;
use DefStudio\GameEngine\Decorators\Tasks\TaskDecorator;
use DefStudio\GameEngine\Models\Concerns\BelongsToScope;
use DefStudio\GameEngine\Models\Concerns\HasStorytellings;
use DefStudio\GameEngine\Models\Contracts\WithStorytelling;

/**
 * @property int $id
 * @property string $title
 * @property int $order
 * @property class-string<TaskDecorator> $type
 * @property ArrayObject<string, mixed> $configuration
 * @property-read Mission $mission
 *
 * @method static TaskFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
class Task extends Model implements HasMedia, WithStorytelling, FitsInScope
{
    use HasFactory;
    use Concerns\HasMedia {
        registerMediaCollections as __registerMediaCollections;
    }
    use HasStorytellings;
    use BelongsToScope;

    public bool $registerMediaConversionsUsingModelInstance = true;

    protected $fillable = [
        'title',
        'type',
    ];

    protected $casts = [
        'configuration' => AsArrayObject::class,
        'order' => 'integer',
    ];

    protected $attributes = [
        'configuration' => '{}',
    ];

    protected array $media_collections = [
        MediaCollection::assets,
        MediaCollection::background,
    ];

    public function getTable()
    {
        return config('game-engine.tasks.table', parent::getTable());
    }

    protected static function newFactory(): TaskFactory
    {
        return TaskFactory::new();
    }

    /**
     * @return BelongsTo<Mission, Task>
     */
    #[ParentScope(Mission::class)]
    public function mission(): BelongsTo
    {
        return $this->belongsTo(game_engine()->missionClass(), 'mission_id');
    }

    public function decorate(): TaskDecorator
    {
        return $this->type::build($this);
    }

    public function registerMediaCollections(): void
    {
        $this->decorate()->__registerMediaCollections();
    }

    public function tags(): array
    {
        return $this->configuration['tags'] ?? [];
    }

    public function has_tag(string $category, string $tag = null): bool
    {
        $key = $tag === null ? $category : "$category:$tag";

        return collect($this->tags())->contains($key);
    }
}
