<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection LaravelUnknownEloquentFactoryInspection */

namespace DefStudio\GameEngine\Models;

use Spatie\MediaLibrary\HasMedia;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Model;
use DefStudio\GameEngine\Enums\MediaCollection;
use DefStudio\GameEngine\Attributes\ParentScope;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Casts\AsArrayObject;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use DefStudio\GameEngine\Models\Contracts\FitsInScope;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use DefStudio\GameEngine\Models\Concerns\DefinesAwards;
use DefStudio\GameEngine\Decorators\Tasks\TaskDecorator;
use DefStudio\GameEngine\Models\Concerns\BelongsToScope;
use DefStudio\GameEngine\Models\Contracts\AwardsDefiner;
use DefStudio\GameEngine\Models\Concerns\HasStorytellings;
use DefStudio\GameEngine\Database\Factories\MissionFactory;
use DefStudio\GameEngine\Models\Contracts\WithStorytelling;
use function game_engine;

/**
 * @property int $id;
 * @property int $map_id;
 * @property string $title;
 * @property string $description;
 * @property int $pos_x;
 * @property int $pos_y;
 * @property int $order;
 * @property array $configuration;
 * @property-read Map $map,
 * @property-read Collection<int, TaskDecorator> $tasks
 *
 * @method static MissionFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
class Mission extends Model implements HasMedia, WithStorytelling, FitsInScope, AwardsDefiner
{
    use HasFactory;
    use Concerns\HasMedia;
    use HasStorytellings;
    use BelongsToScope;
    use DefinesAwards;

    protected $fillable = [
        'title',
        'description',
        'pos_x',
        'pos_y',
        'order',
    ];

    protected $casts = [
        'pos_x' => 'integer',
        'pos_y' => 'integer',
        'order' => 'integer',
        'configuration' => AsArrayObject::class,
    ];

    protected $attributes = [
        'configuration' => '{}',
    ];

    protected array $media_collections = [
        MediaCollection::placeholder,
        MediaCollection::cover,
    ];

    public function getTable()
    {
        return config('game-engine.missions.table', parent::getTable());
    }

    protected static function newFactory(): MissionFactory
    {
        return MissionFactory::new();
    }

    /**
     * @return BelongsTo<Map, Mission>
     */
    #[ParentScope(Map::class)]
    public function map(): BelongsTo
    {
        return $this->belongsTo(game_engine()->mapClass());
    }

    public function tasks(): HasMany
    {
        return $this->hasMany(game_engine()->taskClass());
    }

    /**
     * @return Collection<int, TaskDecorator>
     *
     * @noinspection PhpUnused
     */
    public function getTasksAttribute(): Collection
    {
        return $this->getRelationValue('tasks')
            ->map(function(TaskDecorator|Task $task): TaskDecorator {
                if ($task instanceof TaskDecorator) {
                    return $task;
                }

                return $task->decorate();
            });
    }

    public function award_scope_label(): string
    {
        return __('Mission: :mission_title', ['mission_title' => $this->title]);
    }

    public function max_tasks_to_show(): ?int
    {
        return $this->configuration['max_tasks_to_show'] ?? null;
    }

    public function should_randomize_tasks_order(): bool
    {
        return $this->configuration['randomize_tasks_order'] ?? false;
    }

    public function should_hide_when_not_active(): bool
    {
        return $this->configuration['hide_when_not_active'] ?? false;
    }

    public function should_hide_when_completed(): bool
    {
        return $this->configuration['hide_when_completed'] ?? false;
    }

    public function should_complete_on_activation(): bool
    {
        return $this->configuration['complete_on_activation'] ?? false;
    }

    public function should_autostart_on_activation(): bool
    {
        return $this->configuration['autostart_on_activation'] ?? false;
    }

    public function is_always_active(): bool
    {
        return $this->configuration['always_active'] ?? false;
    }

    public function snapshot(): array
    {
        return [
            'id' => $this->id,
            'title' => $this->title,
            'order' => $this->order,
            'tasks' => $this->tasks
                ->sortBy('order')
                ->map(fn(TaskDecorator $task) => $task->snapshot()),
        ];
    }
}
