<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

// config for DefStudio/GameEngine

use Carbon\CarbonInterval;
use DefStudio\GameEngine\Enums\Role;
use DefStudio\GameEngine\Models\Map;
use DefStudio\GameEngine\Models\Run;
use DefStudio\GameEngine\Models\Task;
use DefStudio\GameEngine\Models\Team;
use DefStudio\GameEngine\Models\Award;
use DefStudio\GameEngine\Models\Story;
use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Models\Mission;
use DefStudio\GameEngine\Models\RunData;
use DefStudio\GameEngine\Decorators\Tasks;
use DefStudio\GameEngine\Enums\Permission;
use DefStudio\GameEngine\Models\Organization;
use DefStudio\GameEngine\Models\Storytelling;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Http\Livewire\Runs\Table;
use DefStudio\GameEngine\Http\Livewire\Awards\Edit;
use DefStudio\GameEngine\Http\Livewire\Maps\Editor;
use DefStudio\GameEngine\Http\Livewire\Runs\Runner;
use DefStudio\GameEngine\Http\Livewire\Stories\Maps;
use DefStudio\GameEngine\Http\Livewire\Awards\AwardList;
use DefStudio\GameEngine\Http\Livewire\Teams\UsersTable;
use DefStudio\GameEngine\Actions\Runs\DefaultLevelEngine;
use DefStudio\GameEngine\Http\Livewire\Runs\MissionModal;
use DefStudio\GameEngine\Actions\Runs\DefaultAwardsEngine;
use DefStudio\GameEngine\Http\Controllers\TeamsController;
use DefStudio\GameEngine\Http\Controllers\UsersController;
use DefStudio\GameEngine\Http\Livewire\Teams\StoriesTable;
use DefStudio\GameEngine\Http\Livewire\Users\ResetPassword;
use DefStudio\GameEngine\Http\Livewire\Storytellings\Manager;
use DefStudio\GameEngine\Http\Livewire\Organizations\Register;
use DefStudio\GameEngine\Http\Livewire\Runs\StorytellingModal;
use DefStudio\GameEngine\Http\Controllers\OrganizationsController;
use DefStudio\GameEngine\Actions\Missions\TaskExtractors\DefaultTaskExtractor;

return [
    'route_prefix' => 'game-engine',
    'users' => [
        'morph_class' => 'user',
        'assign_password_on_creation' => true,
        'reset_password' => [
            'uri' => 'game-engine/reset-password/{user}',
            'handler' => [UsersController::class, 'reset_password'],
            'url_validity_days' => null,
            'redirect_to_route' => 'game-engine.dashboard',
        ],
        'roles' => [
            Role::super_admin->value => [
                'color' => 'bg-orange-500',
                'without_email' => false,
            ],
            Role::admin->value => [
                'color' => 'bg-red-500',
                'without_email' => false,
                'permissions' => [
                    Permission::view_users->for_role(Role::admin),
                    Permission::view_users->for_role(Role::master),
                    Permission::view_users->for_role(Role::player),
                    Permission::create_users->for_role(Role::master),
                    Permission::create_users->for_role(Role::player),
                    Permission::edit_users->for_role(Role::master),
                    Permission::edit_users->for_role(Role::player),
                    Permission::delete_users->for_role(Role::master),
                    Permission::delete_users->for_role(Role::player),

                    Permission::view_organizations->value,
                    Permission::edit_organizations->value,

                    Permission::view_teams->value,
                    Permission::create_teams->value,
                    Permission::edit_teams->value,
                    Permission::subscribe_to_teams->for_role(Role::master),
                    Permission::subscribe_to_teams->for_role(Role::player),
                    Permission::delete_teams->value,

                    Permission::view_stories->value,
                    Permission::create_stories->value,
                    Permission::edit_stories->value,
                    Permission::delete_stories->value,

                    Permission::manage_awards->value,

                    Permission::view_runs->value,
                    Permission::update_runs->value,
                    Permission::delete_runs->value,
                    Permission::debug_runs->value,
                ],
            ],
            Role::master->value => [
                'color' => 'bg-blue-500',
                'without_email' => false,
                'permissions' => [
                    Permission::view_teams->value,
                    Permission::subscribe_to_teams->for_role(Role::player),

                    Permission::view_stories->value,

                    Permission::view_runs->value,
                ],
            ],
            Role::player->value => [
                'color' => 'bg-purple-500',
                'permissions' => [

                ],
                'without_email' => false,
            ],
        ],
        'dashboard_route' => 'game-engine.dashboard',
    ],
    'organizations' => [
        'model' => Organization::class,
        'table' => 'organizations',
        'subscriptions_table' => 'organization_user',
        'morph_class' => 'organization',
        'register_organization' => [
            'uri' => 'game-engine/register_organization',
            'handler' => [OrganizationsController::class, 'register'],
            'url_validity_days' => null,
            'redirect_to_route' => 'game-engine.dashboard',
        ],
    ],
    'teams' => [
        'model' => Team::class,
        'table' => 'teams',
        'subscriptions_table' => 'team_user',
        'morph_class' => 'team',
        'register_team' => [
            'uri' => 'game-engine/organizations/{organization}/register_team',
            'handler' => [TeamsController::class, 'register'],
            'url_validity_days' => null,
            'redirect_to_route' => 'game-engine.dashboard',
        ],
    ],
    'stories' => [
        'model' => Story::class,
        'table' => 'stories',
        'subscriptions_table' => 'story_subscriptions',
        'morph_class' => 'story',
    ],
    'maps' => [
        'model' => Map::class,
        'table' => 'maps',
        'morph_class' => 'map',
    ],
    'missions' => [
        'model' => Mission::class,
        'table' => 'missions',
        'morph_class' => 'mission',
        'task_extractor' => DefaultTaskExtractor::class,
        'modal_width' => '4xl',
    ],
    'tasks' => [
        'model' => Task::class,
        'table' => 'tasks',
        'morph_class' => 'task',
        'assign_awards' => true,
        'types' => [
            Tasks\MultipleChoiceQuestion::class,
            Tasks\DragAndDrop::class,
        ],
    ],
    'storytelling' => [
        'model' => Storytelling::class,
        'table' => 'storytellings',
        'morph_class' => 'storytelling',
        'types' => [
            StorytellingType::activation,
            StorytellingType::prologue,
            StorytellingType::epilogue,
            StorytellingType::timeout,
            StorytellingType::pause,
        ],
        'available_in' => [
            Story::class,
            Map::class,
            Mission::class,
            // Task::class,
        ],
    ],
    'awards' => [
        'model' => Award::class,
        'table' => 'awards',
        'morph_class' => 'award',
        'scopes' => [
            Organization::class,
            Story::class,
            Map::class,
            Mission::class,
        ],
    ],
    'runs' => [
        'model' => Run::class,
        'morph_class' => 'run',
        'table' => 'runs',
        'awards_engine' => DefaultAwardsEngine::class,
        'level_engine' => DefaultLevelEngine::class,
        'timers' => [
            'max_tick_duration_seconds' => CarbonInterval::minutes(10)->totalSeconds,
        ],
        'data' => [
            'model' => RunData::class,
            'morph_class' => 'run_data',
            'table' => 'run_data',
        ],
    ],
    'features' => [
        //Users
        Feature::teams,
        Feature::multi_language,
        Feature::impersonate,
        Feature::welcome_email,
        //Feature::reset_password,
        //Feature::organization_admin_registration_url,
        //Feature::team_master_registration_url,

        //Stories
        Feature::global_stories,
        Feature::storytelling,
        Feature::tag_system,
        Feature::user_stories,
        Feature::team_stories,
        Feature::allow_replay,
        Feature::deadlines,
        Feature::preload_images,
        //Feature::allow_close_missions,
        //Feature::pulse_active_missions,
        //Feature::bounce_active_missions,

        //Awards
        Feature::levels_system,
        Feature::awards_system,
        Feature::awards_hierarchy,
        Feature::scoped_awards,
    ],
    'assets' => [
        'admin' => [
            'js' => [
                'vendor/game-engine/tasks.js',
                'vendor/game-engine/ckeditor.js',
                'vendor/game-engine/livewire-sortable.js',
            ],
            'css' => [
                'vendor/game-engine/ckeditor.css',
                'vendor/game-engine/styles.css',
            ],
        ],
        'runner' => [
            'js' => [
                'vendor/game-engine/runner.js',
                'vendor/game-engine/tasks.js',
                'vendor/game-engine/drag_and_drop_touch_polyfill.js',
            ],
            'css' => [
                'vendor/game-engine/ckeditor.css',
                'vendor/game-engine/styles.css',
            ],
        ],
    ],
    'livewire' => [
        'components' => [
            'users' => [
                'reset_password' => ResetPassword::class,
                'edit' => DefStudio\GameEngine\Http\Livewire\Users\Edit::class,
                'table' => DefStudio\GameEngine\Http\Livewire\Users\Table::class,
                'stories-table' => DefStudio\GameEngine\Http\Livewire\Users\StoriesTable::class,
            ],
            'organizations' => [
                'register' => Register::class,
                'table' => DefStudio\GameEngine\Http\Livewire\Organizations\Table::class,
                'edit' => DefStudio\GameEngine\Http\Livewire\Organizations\Edit::class,
                'users-table' => DefStudio\GameEngine\Http\Livewire\Organizations\UsersTable::class,
            ],
            'teams' => [
                'register' => DefStudio\GameEngine\Http\Livewire\Teams\Register::class,
                'table' => DefStudio\GameEngine\Http\Livewire\Teams\Table::class,
                'edit' => DefStudio\GameEngine\Http\Livewire\Teams\Edit::class,
                'users-table' => UsersTable::class,
                'stories-table' => StoriesTable::class,
            ],
            'stories' => [
                'table' => DefStudio\GameEngine\Http\Livewire\Stories\Table::class,
                'edit' => DefStudio\GameEngine\Http\Livewire\Stories\Edit::class,
                'maps' => Maps::class,
            ],
            'maps' => [
                'edit' => DefStudio\GameEngine\Http\Livewire\Maps\Edit::class,
                'editor' => Editor::class,
            ],
            'missions' => [
                'edit' => DefStudio\GameEngine\Http\Livewire\Missions\Edit::class,
            ],
            'tasks' => [
                'edit' => DefStudio\GameEngine\Http\Livewire\Tasks\Edit::class,
            ],
            'storytelling' => [
                'manager' => Manager::class,
                'edit' => DefStudio\GameEngine\Http\Livewire\Storytellings\Edit::class,
            ],
            'awards' => [
                'list' => AwardList::class,
                'edit' => Edit::class,
            ],
            'runs' => [
                'table' => Table::class,
                'view' => DefStudio\GameEngine\Http\Livewire\Runs\View::class,
                'runner' => [
                    'main' => Runner::class,
                    'mission' => MissionModal::class,
                    'storytelling' => StorytellingModal::class,
                ],
            ],
        ],
    ],

    /*
     * To enable Microsoft Clarity integration, add a new clarity project
     * in https://clarity.microsoft.com/ and copy/paste here the project ID
     * from the page URL: ex https://clarity.microsoft.com/projects/view/<PROJECT_ID>/
     */
    'clarity_integration' => env('GAME_ENGINE_CLARITY_PROJECT_ID'),
];
