<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Actions\Runs\CompleteRun;
use DefStudio\GameEngine\Decorators\Runs\PlayableStory;
use DefStudio\GameEngine\Decorators\Runs\PlayableActivity;

trait RunsStory
{
    public function mountRunsStory(): void
    {
        $this->story->setup_storytelling();
    }

    protected function complete_storytelling_runs_story(PlayableActivity $activity, StorytellingType $type): void
    {
        if (!($activity instanceof PlayableStory)) {
            return;
        }

        if ($type === StorytellingType::activation) {
            $this->start();

            return;
        }

        if ($type === StorytellingType::prologue) {
            $this->start();

            return;
        }

        if ($type === StorytellingType::pause) {
            $this->start();

            return;
        }

        if ($type === StorytellingType::epilogue) {
            $this->complete_story();
        }
    }

    public function start(): void
    {
        if (!$this->story->get_state('setup_completed')) {
            $this->run->started_at = now();

            if (Feature::levels_system->enabled()) {
                game_engine()->level_engine()->compute_starting_level($this->story);
            }

            if (Feature::awards_system->enabled()) {
                game_engine()->awards_engine()->compute_starting_awards($this->story);
            }

            $this->story->set_state('setup_completed', true);
            $this->save();
        }

        if ($this->is_map_active()) {
            $this->story->start_timer();
            $this->save();

            $this->call_traits_method('start');

            return;
        }

        if ($this->show_storytelling($this->story, StorytellingType::activation, StorytellingType::prologue)) {
            return;
        }

        if ($this->show_storytelling($this->story, StorytellingType::pause)) {
            $this->story->pause();

            return;
        }

        $this->story->start_timer();
        $this->save();

        $this->next_map();
    }

    protected function complete_story(): void
    {
        if ($this->show_storytelling($this->story, StorytellingType::epilogue)) {
            return;
        }

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->assign_awards($this->story);
        }

        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->assign_level($this->story);
        }

        $this->story->stop_timer();

        CompleteRun::run($this->run);

        $this->emit('story:completed');

        $this->redirect(route(config('game-engine.users.dashboard_route', 'game-engine.dashboard')));
    }
}
