<?php

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Teams;

use DefStudio\GameEngine\Enums\Role;
use DefStudio\GameEngine\Models\Team;
use DefStudio\GameEngine\Models\User;
use DefStudio\WiredTables\WiredTable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use DefStudio\WiredTables\Configurations\TableConfiguration;

class UsersTable extends WiredTable
{
    /** @var Team */
    public $team;

    public ?string $role_under_creation = null;

    public $listeners = [
        'user.created' => 'handle_user_created',
        'user.updated' => '$refresh',
        'user.switch' => 'handle_user_switch',
    ];

    protected function configure(TableConfiguration $configuration): void
    {
        $configuration
            ->footerView('game-engine::livewire.admin.teams.users-table.footer')
            ->pageSize(10)
            ->groupFilters(false);
    }

    protected function filters(): void
    {
        $this->filter(trans_choice('Role|Roles', 1), 'role')
            ->select(collect([Role::master, Role::player])->mapWithKeys(fn(Role $role): array => [$role->value => $role->label(2)]))
            ->placeholder(__('All Roles'))
            ->handle(fn(BelongsToMany $query, string $role) => $query->wherePivot('role', $role));

        if (game_engine()->user()->can('viewAny', \App\Models\User::class)) {
            $this->filter(__('Display'), 'display')
                ->select(['subscribed' => __('Display Subscribed')])
                ->placeholder(__('Display All Users'))
                ->handle(fn(BelongsToMany $query) => $query->whereHas('teams', fn(Builder $subquery) => $subquery->where('teams.id', $this->team->id)));
        }
    }

    protected function query(): Builder|Relation
    {
        $query = game_engine()->user()->can('viewAny', \App\Models\User::class)
            ? $this->team->organization->users()->wherePivot('role', '!=', Role::admin)
            : $this->team->organization->users()->wherePivot('role', '!=', Role::admin)
                ->whereRelation('teams', 'id', $this->team->id);

        return $query
            ->with(['teams', 'roles']);
    }

    protected function columns(): void
    {
        $this->column(__('Name'), 'name')->searchable()->sortable();
        $this->column(__('Email'), 'email')->searchable()->sortable();
        $this->column(trans_choice('Role|Roles', 2))->view('game-engine::livewire.admin.teams.users-table.role', ['team' => $this->team]);
        $this->column(trans_choice('Action|Actions', 2))->view('game-engine::livewire.admin.teams.users-table.actions', ['team' => $this->team]);
    }

    public function subscribe(int $user_id): void
    {
        /** @var User $user */
        $user = $this->team->organization->users()->findOrFail($user_id);
        $user->subscribe_to_team($this->team);
    }

    public function unsubscribe(int $user_id): void
    {
        /** @var User $user */
        $user = $this->team->organization->users()->findOrFail($user_id);
        $user->unsubscribe_from_team($this->team);
    }

    public function handle_user_created(int $user_id): void
    {
        /** @var User $user */
        $user = game_engine()->userQuery()->find($user_id);

        $role = Role::from($this->role_under_creation);

        $user->subscribe_to_organization($this->team->organization, $role, $user->organizations()->count() === 0);
        $user->subscribe_to_team($this->team);
        $this->emit('user.subscribed.team', $user->id, $this->team->id);
    }

    public function handle_user_switch(int $user_id): void
    {
        /** @var User $user */
        $user = game_engine()->userQuery()->find($user_id);

        $role = Role::from($this->role_under_creation);

        $user->subscribe_to_organization($this->team->organization, $role, $user->organizations()->count() === 0);
        $user->subscribe_to_team($this->team);
        $this->emit('user.subscribed.team', $user->id, $this->team->id);
    }
}
