<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are prohibited without the prior written permission of DEF STUDIO S.R.L. This software is provided "as is" and any express or implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
 */

/** @noinspection PhpUndefinedMethodInspection */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Stories;

use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Contracts\View\View;
use Livewire\TemporaryUploadedFile;
use Illuminate\Support\Facades\Gate;
use Illuminate\Validation\Rules\File;
use DefStudio\GameEngine\Models\Story;
use Illuminate\Validation\ValidationException;
use DefStudio\GameEngine\Actions\Stories\SaveStory;
use DefStudio\GameEngine\Actions\Stories\StoreCoverImage;
use DefStudio\GameEngine\Actions\Stories\RemoveCoverImage;
use DefStudio\GameEngine\Http\Livewire\Concerns\RedirectsBack;

class Edit extends Component
{
    use RedirectsBack;
    use WithFileUploads;

    /** @var Story */
    public $story;

    /** @var TemporaryUploadedFile|null */
    public $cover_image = null;

    public bool $published;

    public bool $global = false;

    public bool $editing;

    public function mount(): void
    {
        Gate::authorize('save', $this->story);
        $this->published = $this->story->published_at !== null;
        $this->editing = !$this->story->exists;
        $this->global = !$this->story->organization_id;
    }

    public function rules(): array
    {
        return [
            'story.title' => ['required', 'string', 'max:255'],
            'story.description' => ['nullable', 'string'],
            'published' => ['nullable', 'boolean'],
        ];
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.stories.edit');
    }

    public function updatedCoverImage(): void
    {
        if ($this->cover_image === null) {
            return;
        }

        try {
            $this->validate(['cover_image' => File::image()]);
        } catch (ValidationException $validationException) {
            $this->cover_image = null;
            throw $validationException;
        }
    }

    public function remove_cover_image(): void
    {
        $this->cover_image = null;
        RemoveCoverImage::run($this->story);
    }

    public function save(): void
    {
        $this->validate();

        if ($this->published && $this->story->published_at === null) {
            $this->story->published_at = now();
        }

        if ($this->cover_image !== null) {
            StoreCoverImage::run($this->story, $this->cover_image);
        }

        SaveStory::run($this->story, $this->global);

        if ($this->story->wasRecentlyCreated) {
            $this->reflash_back_url();
            $this->redirect(route('game-engine.admin.stories.edit', $this->story));

            return;
        }

        $this->editing = false;
    }
}
