<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are prohibited without the prior written permission of DEF STUDIO S.R.L. This software is provided "as is" and any express or implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
 */

namespace DefStudio\GameEngine\Http\Livewire\Runs\Concerns;

use DefStudio\GameEngine\Enums\Feature;
use DefStudio\GameEngine\Enums\StorytellingType;
use DefStudio\GameEngine\Actions\Runs\CompleteRun;
use DefStudio\GameEngine\Decorators\ModelDecorator;
use DefStudio\GameEngine\Decorators\Runs\PlayableStory;
use DefStudio\GameEngine\Models\Contracts\WithStorytelling;

/**
 * @property-read PlayableStory $story
 */
trait RunsStory
{
    public function mountRunsStory(): void
    {
        $this->setup_storytellings($this->story);
    }

    protected function complete_storytelling_runs_story(string|ModelDecorator|WithStorytelling $scope, StorytellingType|string $type): void
    {
        $type = StorytellingType::build_from($type);
        $scope = $this->get_storytelling_scope($scope);

        if ($scope !== 'story') {
            return;
        }

        if ($type === StorytellingType::activation && !$this->show_storytelling($scope, StorytellingType::prologue)) {
            $this->next_map();

            return;
        }

        if ($type === StorytellingType::prologue) {
            $this->next_map();

            return;
        }

        if ($type === StorytellingType::epilogue) {
            $this->complete_story();
        }
    }

    public function getStoryProperty(): PlayableStory
    {
        return PlayableStory::build($this->run->story)->setup($this->run);
    }

    public function start(): void
    {
        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->compute_starting_level($this->story);
        }

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->compute_starting_awards($this->story);
        }

        $this->story->start_timer();
        $this->save();

        if ($this->is_map_active()) {
            $this->call_traits_method('start');

            return;
        }

        if ($this->show_storytelling($this->story, StorytellingType::activation, StorytellingType::prologue)) {
            return;
        }

        $this->next_map();
    }

    protected function complete_story(): void
    {
        if ($this->show_storytelling($this->story, StorytellingType::epilogue)) {
            return;
        }

        if (Feature::awards_system->enabled()) {
            game_engine()->awards_engine()->assign_awards($this->story);
        }

        if (Feature::levels_system->enabled()) {
            game_engine()->level_engine()->assign_level($this->story);
        }

        $this->story->stop_timer();

        CompleteRun::run($this->run);

        $this->emit('story:completed');

        $this->redirect(route('game-engine.dashboard'));
    }
}
