<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */ /** @noinspection PhpDocSignatureIsNotCompleteInspection */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Models\Concerns;

use DefStudio\GameEngine\Models\Team;
use Illuminate\Database\Eloquent\Collection;
use DefStudio\GameEngine\Exceptions\TeamException;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use DefStudio\GameEngine\Events\Teams\UserSubscribedToTeam;
use DefStudio\GameEngine\Events\Teams\UserUnsubscribedFromTeam;
use function game_engine;

trait HasTeams
{
    /**
     * @return BelongsToMany<Team>
     */
    public function teams(): BelongsToMany
    {
        return $this->belongsToMany(game_engine()->teamClass(), config('game-engine.teams.subscriptions_table'))
            ->withTimestamps();
    }

    /**
     * @return BelongsToMany<Team>
     */
    public function current_teams(): BelongsToMany
    {
        /** @phpstan-ignore-next-line */
        return $this->belongsToMany(game_engine()->teamClass())
            ->where('organization_id', $this->current_organization_id)
            ->withTimestamps();
    }

    public function belongs_to_team(Team $team): bool
    {
        return $this->has_attached($team);
    }

    /**
     * @param  Collection<int, Team>  $teams
     */
    public function belongs_to_any_team(Collection $teams): bool
    {
        return $this->has_some_attached($teams);
    }

    public function subscribe_to_team(Team $team): self
    {
        $this->ensure_belongs_to_organization($team->organization);

        throw_if($this->belongs_to_team($team), TeamException::already_subscribed());

        $this->teams()->attach($team);

        UserSubscribedToTeam::dispatch($this, $team);

        return $this;
    }

    public function unsubscribe_from_team(Team $team): self
    {
        throw_if(!$this->belongs_to_team($team), TeamException::team_membership_failure());

        $this->teams()->detach($team);

        UserUnsubscribedFromTeam::dispatch($this, $team);

        return $this;
    }
}
