<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are prohibited without the prior written permission of DEF STUDIO S.R.L. This software is provided "as is" and any express or implied warranties, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable for any direct, indirect, incidental, special, exemplary, or consequential damages (including, but not limited to, procurement of substitute goods or services; loss of use, data, or profits; or business interruption) however caused and on any theory of liability, whether in contract, strict liability, or tort (including negligence or otherwise) arising in any way out of the use of this software, even if advised of the possibility of such damage.
 */ /** @noinspection PhpUnhandledExceptionInspection */

/** @noinspection PhpUnused */

namespace DefStudio\GameEngine\Models;

use DefStudio\GameEngine\Enums\Role;
use DefStudio\GameEngine\Enums\Feature;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use DefStudio\GameEngine\Exceptions\StoryException;
use DefStudio\GameEngine\Models\Concerns\HasStories;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use DefStudio\GameEngine\Database\Factories\TeamFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use DefStudio\GameEngine\Models\Contracts\SubscribesToStories;
use DefStudio\GameEngine\Models\Concerns\CanCheckRelationships;

/**
 * @property int $organization_id
 * @property Collection<int, User> $players
 * @property Collection<int, User> $masters
 *
 * @method static TeamFactory factory(array|callable|int|null $count = null, array|callable $state = [])
 */
class Team extends Model implements SubscribesToStories
{
    use HasFactory;
    use CanCheckRelationships;
    use HasStories;

    protected $fillable = [
        'name',
        'organization_id',
    ];

    public function getTable()
    {
        return config('game-engine.teams.table', parent::getTable());
    }

    protected static function newFactory(): TeamFactory
    {
        return TeamFactory::new();
    }

    /**
     * @return BelongsToMany<User>
     */
    public function players(): BelongsToMany
    {
        /** @phpstan-ignore-next-line */
        return $this->belongsToMany(\game_engine()->userClass(), 'team_user')
            ->whereHas('organizations', function(Builder $query): void {
                $query
                    ->join('teams', 'teams.id', '=', 'team_user.team_id')
                    ->whereRaw('organization_user.organization_id = teams.organization_id')
                    ->where('role', Role::player);
            })
            ->withTimestamps();
    }

    /**
     * @return BelongsToMany<User>
     */
    public function masters(): BelongsToMany
    {
        /** @phpstan-ignore-next-line */
        return $this->belongsToMany(\game_engine()->userClass(), 'team_user')
            ->whereHas('organizations', function(Builder $query): void {
                $query
                    ->join('teams', 'teams.id', '=', 'team_user.team_id')
                    ->whereRaw('organization_user.organization_id = teams.organization_id')
                    ->where('role', Role::master);
            })
            ->withTimestamps();
    }

    /**
     * @return BelongsTo<Organization, Team>
     */
    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    public function stories_feature(): Feature
    {
        return Feature::team_stories;
    }

    public function ensure_can_subscribe_to_story(Story $story): void
    {
        throw_if($story->organization_id !== $this->organization_id, StoryException::organization_membership_failure());
    }
}
