<?php
/*
 * Copyright (c) 2023. DEF STUDIO S.R.L. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are prohibited without the prior written permission of DEF STUDIO S.R.L.
 * This software is provided "as is" and any express or implied warranties, including,
 * but not limited to, the implied warranties of merchantability and fitness for a
 * particular purpose are disclaimed. In no event shall DEF STUDIO S.R.L. be liable
 * for any direct, indirect, incidental, special, exemplary, or consequential damages
 * (including, but not limited to, procurement of substitute goods or services;
 * loss of use, data, or profits; or business interruption) however caused and on any
 * theory of liability, whether in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even if advised of the
 * possibility of such damage.
 */

/** @noinspection PhpUnhandledExceptionInspection */

namespace DefStudio\GameEngine\Http\Livewire\Maps;

use Livewire\WithFileUploads;
use Illuminate\Contracts\View\View;
use Livewire\TemporaryUploadedFile;
use DefStudio\GameEngine\Models\Map;
use Illuminate\Support\Facades\Gate;
use LivewireUI\Modal\ModalComponent;
use Illuminate\Validation\Rules\File;
use DefStudio\GameEngine\Actions\Maps\SaveMap;
use Illuminate\Validation\ValidationException;
use DefStudio\GameEngine\Exceptions\MapException;
use DefStudio\GameEngine\Actions\Maps\StoreBackgroundImage;
use DefStudio\GameEngine\Actions\Maps\RemoveBackgroundImage;
use DefStudio\GameEngine\Http\Livewire\Concerns\ImprovedModal;
use DefStudio\GameEngine\Http\Livewire\Concerns\RedirectsBack;

class Edit extends ModalComponent
{
    use RedirectsBack;
    use WithFileUploads;
    use ImprovedModal;

    /** @var Map */
    public $map;

    /** @var TemporaryUploadedFile|null */
    public $background_image = null;

    public bool $editing;

    public array $rules = [
        'map.story_id' => 'required',
        'map.title' => 'required|string|max:255',
        'map.description' => 'nullable|string',
    ];

    public function mount(int $story_id = null): void
    {
        /** @phpstan-ignore-next-line  */
        throw_if(!isset($this->map) && $story_id === null, MapException::must_be_bound_to_a_story());

        /** @phpstan-ignore-next-line */
        $this->map ??= game_engine()->mapQuery()->make();
        $this->map->story_id ??= $story_id;

        $this->editing = !$this->map->exists;

        Gate::authorize('save', $this->map);
    }

    public function render(): View
    {
        return view('game-engine::livewire.admin.maps.edit');
    }

    public function updatedBackgroundImage(): void
    {
        try {
            $this->validate(['background_image' => File::image()]);
        } catch (ValidationException $validationException) {
            $this->background_image = null;
            throw $validationException;
        }
    }

    public function remove_background_image(): void
    {
        $this->background_image = null;
        RemoveBackgroundImage::run($this->map);
    }

    public function save(): void
    {
        $this->validate();

        SaveMap::run($this->map);

        if ($this->background_image !== null) {
            StoreBackgroundImage::run($this->map, $this->background_image);
        }

        if ($this->map->wasRecentlyCreated) {
            $this->emit('map.created', $this->map->id);
        } else {
            $this->emit('map.updated', $this->map->id);
        }

        $this->emit('map.saved', $this->map->id);

        if ($this->isModal) {
            $this->closeModal();

            return;
        }

        if ($this->map->wasRecentlyCreated) {
            $this->redirect_back();
        } else {
            $this->editing = false;
        }
    }
}
